<?php
defined( 'ABSPATH' ) || exit;

class HRM_Tasks {

    public static function init() {
        add_action( 'admin_post_hrm_save_task', [ __CLASS__, 'handle_save_task' ] );
        add_action( 'wp_ajax_hrm_update_task_status', [ __CLASS__, 'handle_update_task_status' ] );
        add_action( 'wp_ajax_hrm_delete_task', [ __CLASS__, 'handle_ajax_delete_task' ] );
        
        // Employee Frontend CRUD
        add_action( 'wp_ajax_hrm_add_daily_task', [ __CLASS__, 'handle_ajax_save_daily_task' ] );
        add_action( 'wp_ajax_hrm_delete_daily_task', [ __CLASS__, 'handle_ajax_delete_daily_task' ] );
        
        // Admin Ajax
        add_action( 'wp_ajax_hrm_get_progress_details', [ __CLASS__, 'handle_ajax_get_progress_details' ] );
        
        // We will hook frontend actions here if they are standardized, otherwise Portal might call them
        // But for now, we leave the frontend handler public to be called.
    }

    public static function has_submitted_progress( $employee_id, $date ) {
        global $wpdb;
        $count = $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->prefix}hrm_daily_progress WHERE employee_id = %d AND date = %s",
            $employee_id, $date
        ));
        return $count > 0;
    }

    public static function send_daily_progress_email( $employee_id, $date ) {
        global $wpdb;
        $employee_name = get_the_title( $employee_id );
        
        $entries = $wpdb->get_results( $wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}hrm_daily_progress WHERE employee_id = %d AND date = %s",
            $employee_id, $date
        ));

        if ( empty( $entries ) ) {
            error_log("HRM Email Debug: No progress entries found for Employee $employee_id on $date");
            return false;
        }

        $emails_string = get_option( 'hrm_notification_email' );
        if ( empty( $emails_string ) ) {
            $emails_string = get_option( 'admin_email' );
        }
        $emails = array_filter( array_map( 'trim', explode( ',', $emails_string ) ) );
        
        if ( empty( $emails ) ) {
            error_log("HRM Email Debug: No recipient emails found.");
            return false;
        }

        // Ensure email is sent only once per day
        $meta_key = '_hrm_daily_email_sent_' . $date;
        $already_sent = get_user_meta( $employee_id, $meta_key, true );
        
        if ( $already_sent ) {
            error_log("HRM Email Debug: Email already sent to Employee $employee_id for $date. Skipping.");
            return false; // Already sent today
        }
        
        $subject = "Daily Progress Report – " . date_i18n( get_option( 'date_format' ), strtotime( $date ) );
        
        // Fetch Attendance Info
        $attendance = $wpdb->get_row( $wpdb->prepare( 
            "SELECT check_in, check_out FROM {$wpdb->prefix}hrm_attendance WHERE employee_id = %d AND date = %s", 
            $employee_id, $date 
        ));
        
        $check_in_time = ($attendance && $attendance->check_in) ? date('h:i A', strtotime($attendance->check_in)) : 'N/A';
        $check_out_time = ($attendance && $attendance->check_out) ? date('h:i A', strtotime($attendance->check_out)) : 'N/A';
        
        // Calculate Totals (Moved up)
        $total_hours = 0;
        $completed_count = 0;
        $pending_count = 0;
        
        foreach ( $entries as $entry ) {
            $task_status = $wpdb->get_var( $wpdb->prepare( "SELECT status FROM {$wpdb->prefix}hrm_tasks WHERE id = %d", $entry->task_id ) );
            $total_hours += floatval($entry->hours_logged);
            if($task_status === 'completed') $completed_count++; else $pending_count++;
        }
        
        // Create Premium HTML Email
        $message = '<!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <style>
                body { margin: 0; padding: 0; background-color: #ffffff; font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Oxygen, Ubuntu, Cantarell, "Open Sans", "Helvetica Neue", sans-serif; color: #374151; }
                .email-container { max-width: 600px; margin: 20px auto; }
                
                .highlight-box { 
                    background-color: #F0FDF9; 
                    border: 1px solid #E0F2F1; 
                    border-radius: 8px; 
                    padding: 24px; 
                    margin-bottom: 30px; 
                    display: table; /* Fallback for email clients */
                    width: 100%;
                    box-sizing: border-box;
                }
                .highlight-item { 
                    display: table-cell; 
                    vertical-align: middle; 
                    padding-right: 30px;
                }
                .highlight-label { 
                    display: block; 
                    font-size: 11px; 
                    text-transform: uppercase; 
                    color: #6B7280; 
                    font-weight: 700; 
                    letter-spacing: 0.5px; 
                    margin-bottom: 6px; 
                }
                .highlight-value { 
                    display: block; 
                    font-size: 20px; 
                    font-weight: 700; 
                }
                .text-green { color: #10B981; }
                .text-red { color: #EF4444; }
                .text-dark { color: #1F2937; }

                h3.section-title {
                    font-size: 18px;
                    font-weight: 700;
                    color: #1F2937;
                    margin: 0 0 16px 0;
                    padding-left: 12px;
                    border-left: 4px solid #0F766E;
                }

                .task-table { width: 100%; border-collapse: collapse; margin-bottom: 25px; }
                .task-table th { 
                    background-color: #F9FAFB; 
                    color: #6B7280; 
                    font-weight: 700; 
                    text-transform: uppercase; 
                    font-size: 11px; 
                    padding: 12px 16px; 
                    text-align: left; 
                    letter-spacing: 0.5px;
                }
                .task-table td { 
                    padding: 16px; 
                    border-bottom: 1px solid #F3F4F6; 
                    font-size: 14px; 
                    color: #374151; 
                    vertical-align: top; 
                }
                
                .task-title { font-weight: 700; color: #1F2937; display: block; margin-bottom: 4px; }
                .task-desc { color: #6B7280; font-size: 13px; line-height: 1.5; }

                .status-badge { 
                    display: inline-block; 
                    padding: 4px 10px; 
                    border-radius: 6px; 
                    font-size: 11px; 
                    font-weight: 700; 
                    text-transform: uppercase; 
                }
                .status-completed { background-color: #D1FAE5; color: #065F46; } /* Green-100/Green-800 */
                .status-pending { background-color: #F3F4F6; color: #1F2937; }

                .footer { 
                    margin-top: 40px; 
                    text-align: center; 
                    font-size: 12px; 
                    color: #9CA3AF; 
                }

            </style>
        </head>
        <body>
            <div class="email-container">
                <p style="font-size: 16px; margin-bottom: 24px;">Hello Admin,</p>
                <p style="font-size: 16px; margin-bottom: 30px;">Here is the daily work summary for <strong style="color:#111827;">' . esc_html( $employee_name ) . '</strong>.</p>

                <!-- Summary Card -->
                <div class="highlight-box">
                    <div class="highlight-item">
                        <span class="highlight-label">CHECK-IN</span>
                        <span class="highlight-value text-green">' . $check_in_time . '</span>
                    </div>
                    <div class="highlight-item">
                        <span class="highlight-label">CHECK-OUT</span>
                        <span class="highlight-value text-red">' . $check_out_time . '</span>
                    </div>
                    <div class="highlight-item">
                        <span class="highlight-label">TOTAL HOURS</span>
                        <span class="highlight-value text-dark">' . $total_hours . 'h</span>
                    </div>
                </div>

                <h3 class="section-title">Detailed Task Log</h3>
                
                <table class="task-table">
                    <thead>
                        <tr>
                            <th width="30%">TASK</th>
                            <th width="40%">DESCRIPTION</th>
                            <th width="20%">STATUS</th>
                            <th width="10%" style="text-align:center;">HRS</th>
                        </tr>
                    </thead>
                    <tbody>';
        
        foreach ( $entries as $entry ) {
            if ( $entry->task_id == 0 ) {
                // General Work
                $task_title = "General / Other Work";
                $task_status = "In Progress"; // Or 'N/A'
                $status_class = "status-progress"; // neutral
            } else {
                $task = $wpdb->get_row( $wpdb->prepare( "SELECT title, status FROM {$wpdb->prefix}hrm_tasks WHERE id = %d", $entry->task_id ) );
                if(!$task) continue;
                $task_title = $task->title;
                $task_status = ucfirst($task->status);
                $status_class = ($task->status === 'completed') ? 'status-completed' : 'status-progress';
            }
            
            $message .= '<tr>';
            $message .= '<td><strong>' . esc_html( $task_title ) . '</strong></td>';
            $message .= '<td style="color: #64748b; line-height: 1.4;">' . nl2br( esc_html( $entry->description ) ) . '</td>';
            $message .= '<td><span class="status-badge ' . $status_class . '">' . esc_html( $task_status ) . '</span></td>';
            $message .= '<td style="font-weight: 600; text-align: center;">' . esc_html( $entry->hours_logged ) . '</td>';
            $message .= '</tr>';
        }
        
        $message .= '</tbody></table>

                    <!-- Productivity Summary -->
                    <!-- Productivity Summary -->
                    <div style="background-color: #f9fafb; border-radius: 8px; padding: 30px; margin-top: 30px; text-align: center;">
                        <div style="display: inline-block; width: 120px; vertical-align: top; border-right: 1px solid #e5e7eb; padding-right: 20px; margin-right: 20px;">
                            <div style="font-size: 32px; font-weight: 700; color: #065F46; line-height: 1;">' . $completed_count . '</div>
                            <div style="font-size: 10px; font-weight: 700; color: #6B7280; text-transform: uppercase; margin-top: 8px; letter-spacing: 0.5px;">COMPLETED</div>
                        </div>
                        <div style="display: inline-block; width: 120px; vertical-align: top;">
                            <div style="font-size: 32px; font-weight: 700; color: #D97706; line-height: 1;">' . $pending_count . '</div>
                            <div style="font-size: 10px; font-weight: 700; color: #6B7280; text-transform: uppercase; margin-top: 8px; letter-spacing: 0.5px;">IN PROGRESS</div>
                        </div>
                        <div style="margin-top: 20px; color: #9CA3AF; font-size: 20px;">...</div>
                    </div>
                </div>
                
                <div class="footer">
                    &copy; ' . date('Y') . ' HR Management System. All rights reserved.<br>
                    This is an automated notification.
                </div>
            </div>
        </body>
        </html>';

        // Send HTML Email
        add_filter( 'wp_mail_content_type', 'hrm_set_tasks_html_content_type' );
        if( !function_exists('hrm_set_tasks_html_content_type') ) {
            function hrm_set_tasks_html_content_type() { return 'text/html'; }
        }

        $sent_count = 0;
        foreach ( $emails as $email ) {
            if ( is_email( $email ) ) { 
                $result = wp_mail( $email, $subject, $message );
                $result_str = $result ? 'Success' : 'Failed';
                error_log( "HRM Email Debug: Sending to $email... Result: $result_str" );
                if ( $result ) $sent_count++;
            }
        }
        
        // Mark as sent only if successful
        if ( $sent_count > 0 ) {
            update_user_meta( $employee_id, $meta_key, true );
        } else {
             error_log( "HRM Email Debug: Failed to send daily progress email to any recipients." );
        }
        
        remove_filter( 'wp_mail_content_type', 'hrm_set_tasks_html_content_type' );
        return true;
    }

    public static function handle_frontend_actions( $current_employee_id ) {
        if ( ! isset( $_POST['hrm_action'] ) ) return;
        global $wpdb;

        // Progress
        if ( $_POST['hrm_action'] === 'submit_progress' && check_admin_referer( 'hrm_progress_action', 'hrm_progress_nonce' ) ) {
            $employee_id = absint( $_POST['employee_id'] );
            $date = current_time( 'Y-m-d' );
            
            // Validate Tasks or General Progress
            if ( ( empty( $_POST['task_entries'] ) || ! is_array( $_POST['task_entries'] ) ) && empty($_POST['general_progress']) ) {
                wp_die( 'Please select at least one task or enter general progress.' );
            }

            // Calculate Total Hours from Attendance (Source of Truth)
            // Rule: Count from Check-in to Check-out (or Now)
            $attendance = $wpdb->get_row( $wpdb->prepare( 
                "SELECT hours_worked, check_in, check_out FROM {$wpdb->prefix}hrm_attendance WHERE employee_id = %d AND date = %s", 
                $employee_id, $date 
            ));

            $total_hours = 0;
            if ( $attendance ) {
                if ( $attendance->hours_worked > 0 ) {
                    $total_hours = floatval($attendance->hours_worked);
                } elseif ( $attendance->check_in && ! $attendance->check_out ) {
                     // Still checked in: Checkin to Now
                     $start = strtotime( $attendance->check_in );
                     $now = current_time( 'timestamp' );
                     $total_hours = round( ($now - $start) / 3600, 2 );
                }
            }
            
            // Count Entries to Split Hours
            $entry_count = 0;
            $selected_tasks = [];
            
            if ( ! empty( $_POST['task_entries'] ) && is_array( $_POST['task_entries'] ) ) {
                foreach($_POST['task_entries'] as $tid => $tdata) {
                    if(isset($tdata['selected'])) {
                        $selected_tasks[$tid] = $tdata;
                        $entry_count++;
                    }
                }
            }
            if ( ! empty( $_POST['general_progress'] ) ) {
                $entry_count++;
            }
            
            if ( $entry_count == 0 ) wp_die("No entries selected.");

            // Split Hours
            $hours_per_entry = round( $total_hours / $entry_count, 2 );

            // First, clear existing entries for today
            $wpdb->delete( $wpdb->prefix . 'hrm_daily_progress', ['employee_id' => $employee_id, 'date' => $date] );

            // Allowed task statuses for validation
            $allowed_statuses = ['in_process', 'completed'];

            // Insert Tasks AND Update Task Status
            foreach ( $selected_tasks as $task_id => $data ) {
                $desc = sanitize_textarea_field( $data['desc'] );
                if ( empty( $desc ) ) $desc = "Worked on task."; 
                
                // Get task status from form (in_process or completed)
                $task_status = isset($data['task_status']) ? sanitize_key($data['task_status']) : 'in_process';
                
                // Validate status to prevent incorrect values
                if ( ! in_array($task_status, $allowed_statuses) ) {
                    $task_status = 'in_process';
                }
                
                // Verify task ownership before updating (security check)
                $task_owner = $wpdb->get_var( $wpdb->prepare( 
                    "SELECT assigned_to FROM {$wpdb->prefix}hrm_tasks WHERE id = %d", 
                    $task_id 
                ));
                
                // Only update if task belongs to this employee
                if ( (int)$task_owner === (int)$employee_id ) {
                    // Update the main task status in hrm_tasks table
                    $wpdb->update( 
                        $wpdb->prefix . 'hrm_tasks', 
                        ['status' => $task_status], 
                        ['id' => $task_id] 
                    );
                }

                // Insert daily progress record
                $wpdb->insert( $wpdb->prefix . 'hrm_daily_progress', [
                    'employee_id' => $employee_id,
                    'task_id' => $task_id,
                    'date' => $date,
                    'hours_logged' => $hours_per_entry,
                    'description' => $desc,
                    'status' => $task_status
                ]);
            }

            // Insert General Progress
            if ( ! empty( $_POST['general_progress'] ) ) {
                $gen_desc = sanitize_textarea_field( $_POST['general_progress'] );
                
                $wpdb->insert( $wpdb->prefix . 'hrm_daily_progress', [
                    'employee_id' => $employee_id,
                    'task_id' => 0,
                    'date' => $date,
                    'hours_logged' => $hours_per_entry,
                    'description' => $gen_desc,
                    'status' => 'in_progress'
                ]);
            }
            
            // Email will be triggered at Checkout to ensure "once per day" rule.
            // self::send_daily_progress_email( $employee_id, $date );

            wp_redirect( add_query_arg( 'progress_submitted', '1', wp_get_referer() ) );
            exit;
        }

        // Task Completion (Frontend)
        if ( $_POST['hrm_action'] === 'complete_task' && check_admin_referer( 'hrm_task_action' ) ) {
            $task_id = absint( $_POST['task_id'] );
            
            // Verify task belongs to this employee
            $task_owner = $wpdb->get_var( $wpdb->prepare( "SELECT assigned_to FROM {$wpdb->prefix}hrm_tasks WHERE id = %d", $task_id ) );
            if ( (int)$task_owner !== (int)$current_employee_id ) return;
            
            $wpdb->update( $wpdb->prefix . 'hrm_tasks', 
                [ 'status' => 'completed' ],
                [ 'id' => $task_id ]
            );

            wp_redirect( add_query_arg( 'task_completed', '1', wp_get_referer() ) );
            exit;
        }
    }

    public static function render_tasks() {
        if ( isset( $_GET['action'] ) && $_GET['action'] === 'add_new' ) {
            self::render_add_task_page();
            return;
        }
    
        global $wpdb;
        $table = $wpdb->prefix . 'hrm_tasks';
        
        $status_filter = isset($_GET['status']) ? sanitize_text_field($_GET['status']) : '';
        
        // Pagination
        $per_page = HRM_Pagination::get_per_page();
        $current_page = HRM_Pagination::get_current_page();
        $offset = HRM_Pagination::get_offset($current_page, $per_page);
        
        $where_clause = '';
        if ($status_filter) {
            $where_clause = $wpdb->prepare("WHERE status = %s", $status_filter);
        }
        
        // Stats Logic
        $count_pending = $wpdb->get_var( "SELECT COUNT(*) FROM $table WHERE status = 'pending'" );
        $count_completed = $wpdb->get_var( "SELECT COUNT(*) FROM $table WHERE status = 'completed'" );
        $count_in_progress = $wpdb->get_var( "SELECT COUNT(*) FROM $table WHERE status = 'in_progress'" );
        $count_total = $count_pending + $count_completed + $count_in_progress;

        // Total count for pagination (considering filter)
        $total_tasks_filtered = $wpdb->get_var( "SELECT COUNT(*) FROM $table $where_clause" );
        
        $tasks = $wpdb->get_results( $wpdb->prepare(
            "SELECT * FROM $table $where_clause ORDER BY created_at DESC LIMIT %d OFFSET %d",
            $per_page, $offset
        ));
        ?>
        <div class="wrap hrm-wrap hrm-modern-dashboard">
            <!-- Header -->
            <div class="hrm-header-section">
                <div class="hrm-header-left">
                     <h1>Assigned Tasks</h1>
                     <p>Track, manage, and assign work to your team.</p>
                </div>
                <div class="hrm-header-right">
                    <a href="<?php echo admin_url('admin.php?page=hrm-tasks&action=add_new'); ?>" class="hrm-btn-primary">+ Assign Task</a>
                </div>
            </div>

            <!-- Stats Row -->
            <div class="hrm-stats-row">
                <div class="hrm-stat-card white">
                     <div class="stat-header">
                        <span>Pending</span>
                        <span class="dashicons dashicons-hourglass" style="color: var(--warning-color);"></span>
                    </div>
                    <div class="stat-number"><?php echo $count_pending; ?></div>
                    <div class="stat-footer"><span class="text-muted">In Queue</span></div>
                </div>
                <div class="hrm-stat-card white">
                     <div class="stat-header">
                        <span>Completed</span>
                        <span class="dashicons dashicons-yes-alt" style="color: var(--success-color);"></span>
                    </div>
                    <div class="stat-number"><?php echo $count_completed; ?></div>
                    <div class="stat-footer"><span class="text-muted">Finished</span></div>
                </div>
                <div class="hrm-stat-card white">
                     <div class="stat-header">
                        <span>Total Tasks</span>
                        <span class="dashicons dashicons-list-view" style="color: var(--primary-color);"></span>
                    </div>
                    <div class="stat-number"><?php echo $count_total; ?></div>
                    <div class="stat-footer"><span class="text-muted">All Time</span></div>
                </div>
            </div>

            <!-- Filter Bar -->
            <div class="hrm-filter-bar hrm-card" style="margin-bottom: 24px; padding: 10px 20px;">
                <div class="hrm-status-tabs">
                    <a href="<?php echo admin_url('admin.php?page=hrm-tasks'); ?>" class="hrm-tab-link <?php echo !$status_filter ? 'active' : ''; ?>">All Tasks</a>
                    <a href="<?php echo admin_url('admin.php?page=hrm-tasks&status=pending'); ?>" class="hrm-tab-link <?php echo $status_filter==='pending' ? 'active' : ''; ?>">Pending</a>
                    <a href="<?php echo admin_url('admin.php?page=hrm-tasks&status=completed'); ?>" class="hrm-tab-link <?php echo $status_filter==='completed' ? 'active' : ''; ?>">Completed</a>
                </div>
            </div>

            <!-- Task Board Grid -->
            <div class="hrm-task-board" style="display: grid; grid-template-columns: repeat(auto-fill, minmax(300px, 1fr)); gap: 20px;">
                <?php if ( $tasks ) : foreach ( $tasks as $task ) : 
                    $employee_name = get_the_title( $task->assigned_to );
                    $card_border_color = $task->status === 'completed' ? 'var(--success-color)' : 'var(--warning-color)';
                    $status_label = ucfirst($task->status);
                    $status_class = $task->status === 'completed' ? 'badge-green' : 'badge-yellow';
                ?>
                <div class="hrm-card hrm-task-card-modern" style="position: relative; border-left: 4px solid <?php echo $card_border_color; ?>;">
                    <div class="task-card-header" style="display: flex; justify-content: space-between; align-items: flex-start; margin-bottom: 12px;">
                        <span class="hrm-badge <?php echo $status_class; ?>"><?php echo $status_label; ?></span>
                        <div class="task-options" style="display:flex; gap:8px;">
                             <a href="<?php echo admin_url('admin.php?page=hrm-tasks&action=add_new&id=' . $task->id ); ?>" style="text-decoration:none;">
                                <span class="dashicons dashicons-edit" title="Edit" style="color: #6B7280; cursor: pointer;"></span>
                             </a>
                             <span class="dashicons dashicons-trash hrm-delete-task" data-id="<?php echo $task->id; ?>" title="Delete" style="color: #999; cursor: pointer;"></span>
                        </div>
                    </div>

                    <h3 style="margin: 0 0 8px 0; font-size: 16px; font-weight: 600; color: #1F2937;"><?php echo esc_html( $task->title ); ?></h3>
                    <p style="color: #6B7280; font-size: 13px; margin: 0 0 16px 0; line-height: 1.5; min-height: 40px;">
                        <?php echo wp_trim_words( $task->description, 12 ); ?>
                    </p>

                    <div class="task-card-footer" style="display: flex; justify-content: space-between; align-items: center; border-top: 1px solid #F3F4F6; padding-top: 12px; margin-top: auto;">
                        <div class="task-assignee" style="display: flex; align-items: center; gap: 8px;">
                            <div class="hrm-avatar-sm" style="width: 24px; height: 24px; font-size: 10px;"><?php echo strtoupper(substr($employee_name, 0, 1)); ?></div>
                            <span style="font-size: 12px; color: #4B5563; font-weight: 500;"><?php echo esc_html($employee_name); ?></span>
                        </div>
                        <div class="task-due" style="font-size: 12px; color: #6B7280; display:flex; align-items:center; gap:4px;">
                            <span class="dashicons dashicons-calendar" style="font-size: 14px; width: 14px; height: 14px;"></span>
                            <?php echo date('M d', strtotime($task->due_date)); ?>
                        </div>
                    </div>
                </div>
                <?php endforeach; else : ?>
                    <div class="hrm-no-data" style="grid-column: 1/-1;">No tasks found matching your criteria.</div>
                <?php endif; ?>
            </div>
            
            <!-- Pagination -->
            <?php 
            HRM_Pagination::render([
                'total_items' => $total_tasks_filtered,
                'current_page' => $current_page,
                'per_page' => $per_page,
                'query_args' => [
                    'page' => 'hrm-tasks',
                    'status' => $status_filter
                ],
                'show_sizes' => true,
                'context' => 'admin'
            ]);
            HRM_Pagination::render_script();
            ?>
        </div>
        <script>
        jQuery(document).on('click', '.hrm-delete-task', function() {
            if(!confirm('Delete this task?')) return;
            var id = jQuery(this).data('id');
            jQuery.post(ajaxurl, {action: 'hrm_delete_task', task_id: id}, function(res) {
                if(res.success) location.reload();
            });
        });
        jQuery(document).on('click', '.hrm-mark-complete', function() {
            var id = jQuery(this).data('id');
            jQuery.post(ajaxurl, {action: 'hrm_update_task_status', task_id: id, status: 'completed'}, function(res) {
                if(res.success) location.reload();
            });
        });
        </script>
        <?php
    }

    private static function render_add_task_page() {
        global $wpdb;
        $employees = get_posts( [ 'post_type' => HRM_Setup::EMPLOYEE_POST_TYPE, 'numberposts' => -1 ] );
        
        $task_id = isset($_GET['id']) ? absint($_GET['id']) : 0;
        $task = null;
        if ($task_id) {
            $task = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$wpdb->prefix}hrm_tasks WHERE id = %d", $task_id));
        }

        $is_edit = $task ? true : false;
        ?>
        <div class="wrap hrm-wrap hrm-modern-dashboard">
            <div class="hrm-header-actions" style="margin-bottom: 24px;">
                 <a href="<?php echo admin_url('admin.php?page=hrm-tasks'); ?>" class="hrm-back-link">
                    <span class="dashicons dashicons-arrow-left-alt2"></span> Back to Tasks
                 </a>
            </div>
    
            <div class="hrm-form-container-premium">
                <div class="hrm-form-header">
                    <h1><?php echo $is_edit ? 'Edit Task' : 'Assign New Task'; ?></h1>
                    <p><?php echo $is_edit ? 'Update the details for this task.' : 'Create a new task and assign it to a team member.'; ?></p>
                </div>

                <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" class="hrm-premium-form">
                    <input type="hidden" name="action" value="hrm_save_task">
                    <?php if($is_edit): ?>
                        <input type="hidden" name="task_id" value="<?php echo $task->id; ?>">
                    <?php endif; ?>
                    <?php wp_nonce_field( 'hrm_save_task_nonce' ); ?>
                    
                    <div class="hrm-form-section">
                        <h4 class="section-title">Task Details</h4>
                        <div class="hrm-form-group">
                            <label>Task Title <span class="required">*</span></label>
                            <input type="text" name="task_title" required value="<?php echo $is_edit ? esc_attr($task->title) : ''; ?>" placeholder="e.g. Website Redesign - Phase 1">
                        </div>
                        
                        <div class="hrm-row">
                            <div class="hrm-form-group" style="flex:1;">
                                <label>Assign To <span class="required">*</span></label>
                                <select name="assigned_to" required>
                                    <option value="">Select Employee</option>
                                    <?php foreach ( $employees as $emp ) : ?>
                                        <option value="<?php echo $emp->ID; ?>" <?php echo ($is_edit && $task->assigned_to == $emp->ID) ? 'selected' : ''; ?>>
                                            <?php echo esc_html( $emp->post_title ); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="hrm-form-group" style="flex:1;">
                                <label>Due Date <span class="required">*</span></label>
                                <input type="date" name="due_date" required value="<?php echo $is_edit ? esc_attr($task->due_date) : ''; ?>">
                            </div>
                        </div>
                        
                        <?php if($is_edit): ?>
                        <div class="hrm-form-group">
                            <label>Current Status</label>
                            <select name="status">
                                <option value="pending" <?php selected($task->status, 'pending'); ?>>Pending</option>
                                <option value="in_progress" <?php selected($task->status, 'in_progress'); ?>>In Progress</option>
                                <option value="completed" <?php selected($task->status, 'completed'); ?>>Completed</option>
                            </select>
                        </div>
                        <?php endif; ?>

                        <div class="hrm-form-group">
                            <label>Description</label>
                            <textarea name="description" rows="5" placeholder="Enter detailed description, requirements, or notes..." style="width:100%; padding:12px; border-radius:8px; border:1px solid #E5E7EB; font-family:inherit; min-height:100px;"><?php echo $is_edit ? esc_textarea($task->description) : ''; ?></textarea>
                            <span class="hrm-input-hint">Provide clear instructions for the assignee.</span>
                        </div>
                    </div>
                    
                    <div class="hrm-form-actions">
                        <button type="submit" class="hrm-btn-primary hrm-btn-lg"><?php echo $is_edit ? 'Update Task' : 'Assign Task'; ?></button>
                    </div>
                </form>
            </div>
        </div>
        <?php
    }

    public static function handle_save_task() {
        check_admin_referer( 'hrm_save_task_nonce' );
        if ( ! current_user_can( 'manage_options' ) ) wp_die( 'Unauthorized' );
    
        global $wpdb;
        $table = $wpdb->prefix . 'hrm_tasks';
        $employee_id = absint( $_POST['assigned_to'] );
        $title = sanitize_text_field( $_POST['task_title'] );
        $desc = sanitize_textarea_field( $_POST['description'] );
        $due = sanitize_text_field( $_POST['due_date'] );
        
        // --- UPDATE LOGIC ---
        if ( isset( $_POST['task_id'] ) && ! empty( $_POST['task_id'] ) ) {
            $task_id = absint( $_POST['task_id'] );
            $status = sanitize_text_field( $_POST['status'] );
            
            // Get previous state
            $old_task = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM $table WHERE id = %d", $task_id ) );
            if ( ! $old_task ) wp_die("Task not found.");

            $wpdb->update( $table, [
                'title'       => $title,
                'assigned_to' => $employee_id,
                'due_date'    => $due,
                'description' => $desc,
                'status'      => $status
            ], [ 'id' => $task_id ] );
            
            // Notification: If Completed task is updated
            if ( $old_task->status === 'completed' ) {
                $notif_msg = "Your completed task '{$title}' has been updated by Admin. Please review the changes.";
                
                // 1. In-App Notification
                $wpdb->insert( $wpdb->prefix . 'hrm_notifications', [
                    'employee_id' => $employee_id,
                    'type'        => 'task_updated',
                    'message'     => $notif_msg,
                    'related_id'  => $task_id,
                    'is_read'     => 0
                ]);
                
                // 2. Email Notification (Robust Lookup)
                $employee_email = get_post_meta( $employee_id, 'hrm_email', true );
                if ( ! $employee_email ) {
                     // Fallback to Author
                     $employee_email = get_the_author_meta('user_email', get_post_field('post_author', $employee_id));
                }
                
                if ( $employee_email && is_email($employee_email) ) {
                    $employee_name = get_the_title( $employee_id );
                    $portal_url = home_url('/employee-portal/');
                    $subject = "Task Updated: " . $title;
                    
                    $message = '<!DOCTYPE html>
                    <html>
                    <head>
                        <meta charset="UTF-8">
                        <meta name="viewport" content="width=device-width, initial-scale=1.0">
                        <style>
                            /* Responsive and General Styles */
                            body { margin: 0; padding: 0; background-color: #F3F4F6; font-family: "Helvetica Neue", Helvetica, Arial, sans-serif; }
                        </style>
                    </head>
                    <body style="background-color: #F3F4F6; margin: 0; padding: 0; font-family: \'Helvetica Neue\', Helvetica, Arial, sans-serif; color: #374151;">
                        <table width="100%" border="0" cellspacing="0" cellpadding="0" style="background-color: #F3F4F6;">
                            <tr>
                                <td align="center" style="padding: 40px 0;">
                                    <table width="600" border="0" cellspacing="0" cellpadding="0" style="background-color: #ffffff; border-radius: 16px; overflow: hidden; box-shadow: 0 10px 25px -5px rgba(0, 0, 0, 0.1);">
                                        <!-- Header -->
                                        <tr>
                                            <td style="background: linear-gradient(135deg, #0F766E 0%, #0D9488 100%); background-color: #0F766E; padding: 40px 30px; text-align: center;">
                                                <h1 style="color: #ffffff; margin: 0; font-size: 24px; font-weight: 700; letter-spacing: 0.5px;">Task Update</h1>
                                            </td>
                                        </tr>
                                        <!-- Content -->
                                        <tr>
                                            <td style="padding: 40px 30px;">
                                                <p style="font-size: 18px; color: #111827; margin: 0 0 24px 0;">Hello <strong>' . esc_html($employee_name) . '</strong>,</p>
                                                <p style="margin: 0 0 30px 0; font-size: 15px; color: #4B5563; line-height: 1.6;">Your completed task data has been updated by the Admin. Please review the changes below.</p>
                                                
                                                <!-- Card -->
                                                <table width="100%" border="0" cellspacing="0" cellpadding="0" style="background-color: #FFFFFF; border: 1px solid #E5E7EB; border-radius: 12px; margin-bottom: 30px;">
                                                    <tr>
                                                        <td style="background-color: #F9FAFB; padding: 16px 24px; border-bottom: 1px solid #E5E7EB;">
                                                            <span style="height: 8px; width: 8px; background-color: #10B981; border-radius: 50%; display: inline-block; margin-right: 8px;"></span>
                                                            <span style="font-size: 14px; font-weight: 600; color: #6B7280; text-transform: uppercase; letter-spacing: 1px;">Task Details</span>
                                                        </td>
                                                    </tr>
                                                    <tr>
                                                        <td style="padding: 24px;">
                                                            <h2 style="font-size: 22px; font-weight: 800; color: #1F2937; margin: 0 0 12px 0;">' . esc_html($title) . '</h2>
                                                            
                                                            <table width="100%" border="0" cellspacing="0" cellpadding="0" style="margin-bottom: 20px;">
                                                                <tr>
                                                                    <td width="50%" valign="top" style="padding-bottom: 15px;">
                                                                        <span style="font-size: 11px; font-weight: 700; color: #9CA3AF; text-transform: uppercase; display: block; margin-bottom: 4px;">Update Note</span>
                                                                        <span style="font-size: 15px; font-weight: 600; color: #0F766E;">' . wp_trim_words($notif_msg, 10) . '</span>
                                                                    </td>
                                                                </tr>
                                                            </table>

                                                            <span style="font-size: 11px; font-weight: 700; color: #9CA3AF; text-transform: uppercase; display: block; margin-bottom: 4px;">Description</span>
                                                            <div style="background-color: #F8FAFC; border-left: 4px solid #0F766E; padding: 16px; border-radius: 4px; color: #4B5563; font-size: 15px; line-height: 1.6; font-style: italic;">
                                                                ' . nl2br(esc_html($desc)) . '
                                                            </div>
                                                        </td>
                                                    </tr>
                                                </table>
                                                
                                                <div style="text-align: center; margin-top: 40px;">
                                                    <a href="' . esc_attr($portal_url) . '" style="background-color: #0F766E; color: #ffffff; padding: 14px 32px; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 16px; display: inline-block;">View Task in Portal</a>
                                                </div>
                                            </td>
                                        </tr>
                                        <!-- Footer -->
                                        <tr>
                                            <td style="background-color: #F9FAFB; padding: 24px; text-align: center; font-size: 12px; color: #9CA3AF; border-top: 1px solid #F3F4F6;">
                                                <p style="margin: 0 0 8px 0;">&copy; ' . date('Y') . ' HR Management System. All rights reserved.</p>
                                                <p style="margin: 0;">Automated Notification System</p>
                                            </td>
                                        </tr>
                                    </table>
                                </td>
                            </tr>
                        </table>
                    </body>
                    </html>';
                    
                    add_filter( 'wp_mail_content_type', 'hrm_set_html_content_type_task' );
                    if( !function_exists('hrm_set_html_content_type_task') ) { function hrm_set_html_content_type_task() { return 'text/html'; } }
                    wp_mail( $employee_email, $subject, $message );
                    remove_filter( 'wp_mail_content_type', 'hrm_set_html_content_type_task' );
                }
            }
            
            wp_redirect( admin_url( 'admin.php?page=hrm-tasks&message=updated' ) );
            exit;
        }

        // --- INSERT LOGIC (New Task) ---
        $wpdb->insert( $table, [
            'title'       => $title,
            'assigned_to' => $employee_id,
            'due_date'    => $due,
            'description' => $desc,
            'status'      => 'pending'
        ]);
        
        $task_id = $wpdb->insert_id;

        // 1. In-App Notification
        $notif_msg = "A new task '{$title}' has been assigned to you by Admin.";
        $wpdb->insert( $wpdb->prefix . 'hrm_notifications', [
            'employee_id' => $employee_id,
            'type'        => 'task_assigned',
            'message'     => $notif_msg,
            'related_id'  => $task_id,
            'is_read'     => 0
        ]);

        // 2. Email Notification
        // 2. Email Notification (Robust Lookup)
        $employee_email = get_post_meta( $employee_id, 'hrm_email', true );
        if ( ! $employee_email ) {
             // Fallback to Author
             $employee_email = get_the_author_meta('user_email', get_post_field('post_author', $employee_id));
        }

        if ( $employee_email && is_email($employee_email) ) {
            $subject = "✨ New Task Assigned: " . $title;
            
            // Employee Name for personalization
            $employee_name = get_the_title( $employee_id );
            $portal_url = home_url('/employee-portal/');
            
            $message = '<!DOCTYPE html>
            <html>
            <head>
                <meta charset="UTF-8">
                <meta name="viewport" content="width=device-width, initial-scale=1.0">
                <style>
                    body { margin: 0; padding: 0; background-color: #F3F4F6; font-family: "Helvetica Neue", Helvetica, Arial, sans-serif; }
                </style>
            </head>
            <body style="background-color: #F3F4F6; margin: 0; padding: 0; font-family: \'Helvetica Neue\', Helvetica, Arial, sans-serif; color: #374151;">
                <table width="100%" border="0" cellspacing="0" cellpadding="0" style="background-color: #F3F4F6;">
                    <tr>
                        <td align="center" style="padding: 40px 0;">
                            <table width="600" border="0" cellspacing="0" cellpadding="0" style="background-color: #ffffff; border-radius: 16px; overflow: hidden; box-shadow: 0 10px 25px -5px rgba(0, 0, 0, 0.1);">
                                <!-- Header -->
                                <tr>
                                    <td style="background: linear-gradient(135deg, #0F766E 0%, #0D9488 100%); background-color: #0F766E; padding: 40px 30px; text-align: center;">
                                        <h1 style="color: #ffffff; margin: 0; font-size: 24px; font-weight: 700; letter-spacing: 0.5px;">New Task Assignment</h1>
                                    </td>
                                </tr>
                                <!-- Content -->
                                <tr>
                                    <td style="padding: 40px 30px;">
                                        <p style="font-size: 18px; color: #111827; margin: 0 0 24px 0;">Hello <strong>' . esc_html($employee_name) . '</strong>,</p>
                                        <p style="margin: 0 0 30px 0; font-size: 15px; color: #4B5563; line-height: 1.6;">You have been assigned a new task. Please check the details below and ensure timely completion.</p>
                                        
                                        <!-- Card -->
                                        <table width="100%" border="0" cellspacing="0" cellpadding="0" style="background-color: #FFFFFF; border: 1px solid #E5E7EB; border-radius: 12px; margin-bottom: 30px;">
                                            <tr>
                                                <td style="background-color: #F9FAFB; padding: 16px 24px; border-bottom: 1px solid #E5E7EB;">
                                                    <span style="height: 8px; width: 8px; background-color: #F59E0B; border-radius: 50%; display: inline-block; margin-right: 8px;"></span>
                                                    <span style="font-size: 14px; font-weight: 600; color: #6B7280; text-transform: uppercase; letter-spacing: 1px;">Task Details</span>
                                                </td>
                                            </tr>
                                            <tr>
                                                <td style="padding: 24px;">
                                                    <h2 style="font-size: 22px; font-weight: 800; color: #1F2937; margin: 0 0 12px 0;">' . esc_html($title) . '</h2>
                                                    
                                                    <table width="100%" border="0" cellspacing="0" cellpadding="0" style="margin-bottom: 20px;">
                                                        <tr>
                                                            <td width="50%" valign="top" style="padding-bottom: 15px;">
                                                                <span style="font-size: 11px; font-weight: 700; color: #9CA3AF; text-transform: uppercase; display: block; margin-bottom: 4px;">Due Date</span>
                                                                <span style="font-size: 15px; font-weight: 600; color: #DC2626;">' . date_i18n(get_option('date_format'), strtotime($due)) . '</span>
                                                            </td>
                                                            <td width="50%" valign="top" style="padding-bottom: 15px;">
                                                                <span style="font-size: 11px; font-weight: 700; color: #9CA3AF; text-transform: uppercase; display: block; margin-bottom: 4px;">Current Status</span>
                                                                <span style="font-size: 15px; font-weight: 600; color: #1F2937;">Pending</span>
                                                            </td>
                                                        </tr>
                                                    </table>

                                                    <span style="font-size: 11px; font-weight: 700; color: #9CA3AF; text-transform: uppercase; display: block; margin-bottom: 4px;">Description & Instructions</span>
                                                    <div style="background-color: #F8FAFC; border-left: 4px solid #0F766E; padding: 16px; border-radius: 4px; color: #4B5563; font-size: 15px; line-height: 1.6;">
                                                        ' . nl2br(esc_html($desc)) . '
                                                    </div>
                                                </td>
                                            </tr>
                                        </table>
                                        
                                        <div style="text-align: center; margin-top: 40px;">
                                            <a href="' . esc_attr($portal_url) . '" style="background-color: #0F766E; color: #ffffff; padding: 14px 32px; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 16px; display: inline-block;">Access Employee Portal</a>
                                        </div>
                                    </td>
                                </tr>
                                <!-- Footer -->
                                <tr>
                                    <td style="background-color: #F9FAFB; padding: 24px; text-align: center; font-size: 12px; color: #9CA3AF; border-top: 1px solid #F3F4F6;">
                                        <p style="margin: 0 0 8px 0;">&copy; ' . date('Y') . ' HR Management System. All rights reserved.</p>
                                        <p style="margin: 0;">Automated Notification System</p>
                                    </td>
                                </tr>
                            </table>
                        </td>
                    </tr>
                </table>
            </body>
            </html>';
            
            // Send HTML Email
            add_filter( 'wp_mail_content_type', 'hrm_set_html_content_type_task' );
            if( !function_exists('hrm_set_html_content_type_task') ) {
                function hrm_set_html_content_type_task() { return 'text/html'; }
            }
            
            wp_mail( $employee_email, $subject, $message );
            
            remove_filter( 'wp_mail_content_type', 'hrm_set_html_content_type_task' );
        }
    
        wp_redirect( admin_url( 'admin.php?page=hrm-tasks&message=saved' ) );
        exit;
    }

    public static function handle_update_task_status() {
        // Allow generic ajax nonce for frontend usage
        if( isset($_REQUEST['nonce']) && wp_verify_nonce( $_REQUEST['nonce'], 'hrm_ajax_nonce' ) ) {
            // verified
        } else {
            check_ajax_referer( 'hrm_update_status_nonce' );
        }
        // Or if context is mixed.
        // Assuming admin-ajax call from Dashboard.
        
        $task_id = isset($_POST['task_id']) ? intval( $_POST['task_id'] ) : 0;
        $status = sanitize_text_field( $_POST['status'] );
        $progress_id = isset($_POST['progress_id']) ? intval( $_POST['progress_id'] ) : 0;
        $comment = isset($_POST['comment']) ? sanitize_textarea_field( $_POST['comment'] ) : '';

        global $wpdb;
        $table_tasks = $wpdb->prefix . 'hrm_tasks';
        $table_progress = $wpdb->prefix . 'hrm_daily_progress';
        $table_notifications = $wpdb->prefix . 'hrm_notifications';

        if ( $task_id > 0 ) {
            $wpdb->update( $table_tasks, [ 'status' => $status ], [ 'id' => $task_id ] );
        }
        
        // Admin Comment on Progress
        if ( $progress_id > 0 ) {
             $wpdb->update($table_progress, [ 'admin_comment' => $comment ], [ 'id' => $progress_id ]);

             if ( ! empty( $comment ) ) {
                 $progress = $wpdb->get_row( $wpdb->prepare( "SELECT employee_id FROM $table_progress WHERE id = %d", $progress_id ) );
                 if ( $progress ) {
                     $wpdb->insert( $table_notifications, [
                         'employee_id' => $progress->employee_id,
                         'type'        => 'admin_comment',
                         'message'     => $comment,
                         'related_id'  => $progress_id,
                         'is_read'     => 0
                     ]);
                     
                     // Send Email
                     // Send Email (Robust Lookup)
                     $employee_email = get_post_meta( $progress->employee_id, 'hrm_email', true );
                     if ( ! $employee_email ) {
                        $employee_email = get_the_author_meta('user_email', get_post_field('post_author', $progress->employee_id));
                     }

                     if ( $employee_email && is_email($employee_email) ) {
                        $employee_name = get_the_title( $progress->employee_id );
                        $portal_url = home_url('/employee-portal/');
                        $subject = "Admin Comment on Progress";
                        
                        $message = '<!DOCTYPE html>
                        <html>
                        <head>
                            <meta charset="UTF-8">
                            <meta name="viewport" content="width=device-width, initial-scale=1.0">
                            <style>
                                body { margin: 0; padding: 0; background-color: #F3F4F6; font-family: "Helvetica Neue", Helvetica, Arial, sans-serif; }
                            </style>
                        </head>
                        <body style="background-color: #F3F4F6; margin: 0; padding: 0; font-family: \'Helvetica Neue\', Helvetica, Arial, sans-serif; color: #374151;">
                            <table width="100%" border="0" cellspacing="0" cellpadding="0" style="background-color: #F3F4F6;">
                                <tr>
                                    <td align="center" style="padding: 40px 0;">
                                        <table width="600" border="0" cellspacing="0" cellpadding="0" style="background-color: #ffffff; border-radius: 16px; overflow: hidden; box-shadow: 0 10px 25px -5px rgba(0, 0, 0, 0.1);">
                                            <!-- Header -->
                                            <tr>
                                                <td style="background: linear-gradient(135deg, #0F766E 0%, #0D9488 100%); background-color: #0F766E; padding: 40px 30px; text-align: center;">
                                                    <h1 style="color: #ffffff; margin: 0; font-size: 24px; font-weight: 700; letter-spacing: 0.5px;">New Admin Comment</h1>
                                                </td>
                                            </tr>
                                            <!-- Content -->
                                            <tr>
                                                <td style="padding: 40px 30px;">
                                                    <p style="font-size: 18px; color: #111827; margin: 0 0 24px 0;">Hello <strong>' . esc_html($employee_name) . '</strong>,</p>
                                                    <p style="margin: 0 0 30px 0; font-size: 15px; color: #4B5563; line-height: 1.6;">An administrator has added a comment to your daily progress report.</p>
                                                    
                                                    <!-- Card -->
                                                    <table width="100%" border="0" cellspacing="0" cellpadding="0" style="background-color: #FFFFFF; border: 1px solid #E5E7EB; border-radius: 12px; margin-bottom: 30px;">
                                                        <tr>
                                                            <td style="background-color: #F9FAFB; padding: 16px 24px; border-bottom: 1px solid #E5E7EB;">
                                                                <span style="height: 8px; width: 8px; background-color: #3B82F6; border-radius: 50%; display: inline-block; margin-right: 8px;"></span>
                                                                <span style="font-size: 14px; font-weight: 600; color: #6B7280; text-transform: uppercase; letter-spacing: 1px;">Comment Details</span>
                                                            </td>
                                                        </tr>
                                                        <tr>
                                                            <td style="padding: 24px;">
                                                                <span style="font-size: 11px; font-weight: 700; color: #9CA3AF; text-transform: uppercase; display: block; margin-bottom: 4px;">Admin Message</span>
                                                                <div style="background-color: #E0F2FE; border-left: 4px solid #0284C7; padding: 16px; border-radius: 4px; color: #0C4A6E; font-size: 15px; line-height: 1.6;">
                                                                    ' . nl2br(esc_html($comment)) . '
                                                                </div>
                                                            </td>
                                                        </tr>
                                                    </table>
                                                    
                                                    <div style="text-align: center; margin-top: 40px;">
                                                        <a href="' . esc_attr($portal_url) . '" style="background-color: #0F766E; color: #ffffff; padding: 14px 32px; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 16px; display: inline-block;">View Progress Report</a>
                                                    </div>
                                                </td>
                                            </tr>
                                            <!-- Footer -->
                                            <tr>
                                                <td style="background-color: #F9FAFB; padding: 24px; text-align: center; font-size: 12px; color: #9CA3AF; border-top: 1px solid #F3F4F6;">
                                                    <p style="margin: 0 0 8px 0;">&copy; ' . date('Y') . ' HR Management System. All rights reserved.</p>
                                                    <p style="margin: 0;">Automated Notification System</p>
                                                </td>
                                            </tr>
                                        </table>
                                    </td>
                                </tr>
                            </table>
                        </body>
                        </html>';
                        
                        add_filter( 'wp_mail_content_type', 'hrm_set_html_content_type_task' );
                        if( !function_exists('hrm_set_html_content_type_task') ) { function hrm_set_html_content_type_task() { return 'text/html'; } }
                        wp_mail( $employee_email, $subject, $message );
                        remove_filter( 'wp_mail_content_type', 'hrm_set_html_content_type_task' );
                     }
                 }
             }
        }
        
        wp_send_json_success();
    }

    public static function handle_ajax_delete_task() {
        ob_clean();
        if ( ! current_user_can( 'manage_options' ) ) wp_send_json_error( 'Unauthorized' );
    
        $task_id = absint( $_POST['task_id'] );
        global $wpdb;
        $deleted = $wpdb->delete( $wpdb->prefix . 'hrm_tasks', ['id' => $task_id] );
    
        if ( $deleted === false ) wp_send_json_error( 'Failed to delete task.' );
        wp_send_json_success( 'Task deleted successfully.' );
    }

    public static function handle_ajax_get_progress_details() {
        check_ajax_referer('hrm_ajax_nonce', 'nonce');
        if ( ! current_user_can('manage_options') ) wp_send_json_error('Unauthorized');

        global $wpdb;
        $progress_id = intval($_POST['id']);

        $sql = "SELECT 
                    p.date, p.hours_logged, p.description as progress_desc, p.admin_comment,
                    t.task_id as id, t.title as task_title, t.description as task_desc, t.status as task_status,
                    e.post_title as employee_name
                FROM {$wpdb->prefix}hrm_daily_progress p
                LEFT JOIN {$wpdb->prefix}hrm_tasks t ON p.task_id = t.id
                LEFT JOIN {$wpdb->posts} e ON p.employee_id = e.ID
                WHERE p.id = %d";
        
        // Note: The above LEFT JOIN t.task_id is wrong, t.id is the column. 
        // Correcting: t.id as task_id_pk
        $sql = "SELECT 
                    p.date, p.hours_logged, p.description as progress_desc, p.admin_comment,
                    t.id as task_id_pk, t.title as task_title, t.description as task_desc, t.status as task_status,
                    e.post_title as employee_name
                FROM {$wpdb->prefix}hrm_daily_progress p
                LEFT JOIN {$wpdb->prefix}hrm_tasks t ON p.task_id = t.id
                LEFT JOIN {$wpdb->posts} e ON p.employee_id = e.ID
                WHERE p.id = %d";

        $data = $wpdb->get_row($wpdb->prepare($sql, $progress_id));
        
        if ( ! $data ) wp_send_json_error('Data not found');

        // Format Date
        $data->date = date('M d, Y', strtotime($data->date));
        
        wp_send_json_success($data);
    }

    public static function handle_ajax_save_daily_task() {
        check_ajax_referer( 'hrm_ajax_nonce', 'nonce' );
        
        $current_user = wp_get_current_user();
        // Robustly Find Employee ID
        $employee_id = 0;
        
        // 1. Try by Author
        $employee_posts = get_posts([
            'post_type' => HRM_Setup::EMPLOYEE_POST_TYPE,
            'author'    => $current_user->ID,
            'numberposts' => 1,
            'fields'    => 'ids'
        ]);
        
        if ( ! empty( $employee_posts ) ) {
             $employee_id = $employee_posts[0];
        } else {
             // 2. Try by Email Meta
             $employee_posts_by_email = get_posts([
                'post_type'  => HRM_Setup::EMPLOYEE_POST_TYPE,
                'meta_key'   => 'hrm_email',
                'meta_value' => $current_user->user_email,
                'numberposts' => 1,
                'fields'     => 'ids'
            ]);
            if ( ! empty( $employee_posts_by_email ) ) {
                $employee_id = $employee_posts_by_email[0];
            }
        }

        if ( ! $employee_id ) wp_send_json_error('Employee profile not found for your account.');

        $task_id = isset($_POST['task_id']) ? absint($_POST['task_id']) : 0;
        $title = sanitize_text_field($_POST['title']);
        $description = sanitize_textarea_field($_POST['description']);
        $due_date = sanitize_text_field($_POST['due_date']);
        $status = isset($_POST['status']) ? sanitize_key($_POST['status']) : 'pending';

        if(empty($title)) wp_send_json_error('Task title is required.');
        if(empty($due_date)) $due_date = current_time('Y-m-d'); // Fallback

        global $wpdb;
        $table = $wpdb->prefix . 'hrm_tasks';
        
        // Log incoming data
        error_log("HRM Attempting Save Task: " . json_encode($_POST));

        $data = [
            'title' => $title,
            'description' => $description,
            'due_date' => $due_date,
            'status' => $status,
            'assigned_to' => $employee_id
        ];

        if($task_id > 0) {
            // Edit Mode
            $updated = $wpdb->update($table, $data, ['id' => $task_id]);
            if($updated === false) {
                 error_log("HRM Update Failed: " . $wpdb->last_error);
                 wp_send_json_error("Update failed: " . $wpdb->last_error);
            }
            wp_send_json_success('Task updated successfully.');
        } else {
            // Add Mode
            $result = $wpdb->insert($table, $data);
            
            if ( $result === false ) {
                $log_file = HRM_PLUGIN_PATH . 'hrm_debug.log';
                error_log( date('[Y-m-d H:i:s] ') . "HRM Error: Failed to insert task. DB Error: " . $wpdb->last_error . " | Data: " . json_encode($data) . "\n", 3, $log_file );
                wp_send_json_error('Database error: ' . $wpdb->last_error);
            }
            
            $log_file = HRM_PLUGIN_PATH . 'hrm_debug.log';
            error_log( date('[Y-m-d H:i:s] ') . "HRM Task Created: " . json_encode($data) . " | ID: " . $wpdb->insert_id . " | Owner: $employee_id\n", 3, $log_file );
            wp_send_json_success('Task created successfully.');
        }
    }

    public static function handle_ajax_delete_daily_task() {
        check_ajax_referer( 'hrm_ajax_nonce', 'nonce' );
        
        $current_user = wp_get_current_user();
        
        // Robustly find Employee ID
        // 1. Try by Author
        $employee_posts = get_posts([
            'post_type' => HRM_Setup::EMPLOYEE_POST_TYPE,
            'author'    => $current_user->ID,
            'numberposts' => 1,
            'fields'    => 'ids'
        ]);
        
        $employee_id = 0;
        if ( ! empty( $employee_posts ) ) {
            $employee_id = $employee_posts[0];
        } else {
            // 2. Try by Email Meta
            $employee_posts_by_email = get_posts([
                'post_type'  => HRM_Setup::EMPLOYEE_POST_TYPE,
                'meta_key'   => 'hrm_email',
                'meta_value' => $current_user->user_email,
                'numberposts' => 1,
                'fields'     => 'ids'
            ]);
            if ( ! empty( $employee_posts_by_email ) ) {
                $employee_id = $employee_posts_by_email[0];
            }
        }

        if ( ! $employee_id ) wp_send_json_error('Employee profile not found for your account.');
        
        $task_id = absint($_POST['task_id']);
        global $wpdb;
        $table = $wpdb->prefix . 'hrm_tasks';

        // Check Ownership (Assigned To)
        $owner = $wpdb->get_var($wpdb->prepare("SELECT assigned_to FROM $table WHERE id = %d", $task_id));
        
        if ( (int)$owner !== (int)$employee_id ) {
             wp_send_json_error('Unauthorized: You are not assigned to this task.');
        }

        $wpdb->delete($table, ['id' => $task_id]);
        wp_send_json_success('Task deleted successfully.');
    }

    public static function render_frontend_tasks( $employee_id ) {
        if ( ! $employee_id ) {
             echo '<p class="hrm-content-notice error">System Error: Employee ID missing.</p>';
             return;
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'hrm_tasks';
        $tasks = $wpdb->get_results( $wpdb->prepare( 
            "SELECT * FROM $table_name WHERE assigned_to = %d ORDER BY status ASC, due_date ASC", 
            $employee_id 
        ));
         
        // Fallback for debugging if query fails silently
        if ( $tasks === null ) {
            echo '<p class="hrm-content-notice error">Database Error: ' . $wpdb->last_error . '</p>';
            return;
        }
        ?>
        <style>
            .hrm-tasks-list { font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Oxygen, Ubuntu, Cantarell, "Open Sans", "Helvetica Neue", sans-serif; }
            
            /* Animations */
            @keyframes fadeInUp {
                from { opacity: 0; transform: translate3d(0, 20px, 0); }
                to { opacity: 1; transform: translate3d(0, 0, 0); }
            }
            @keyframes shimmer {
                0% { background-position: -200% 0; }
                100% { background-position: 200% 0; }
            }

            .hrm-task-card-list {
                perspective: 1000px;
            }
            
            .hrm-frontend-task-card {
                background: #fff;
                border: 1px solid #E5E7EB;
                border-radius: 16px;
                padding: 24px;
                margin-bottom: 20px;
                display: flex;
                justify-content: space-between;
                align-items: flex-start;
                box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.05), 0 2px 4px -1px rgba(0, 0, 0, 0.03);
                transition: transform 0.3s ease, box-shadow 0.3s ease, border-color 0.3s ease;
                
                /* Entrance Animation */
                animation: fadeInUp 0.5s ease-out forwards;
                opacity: 0; /* Starts hidden */
                position: relative;
                overflow: hidden;
            }

            /* Staggered Delay for up to 10 items */
            .hrm-frontend-task-card:nth-child(1) { animation-delay: 0.1s; }
            .hrm-frontend-task-card:nth-child(2) { animation-delay: 0.2s; }
            .hrm-frontend-task-card:nth-child(3) { animation-delay: 0.3s; }
            .hrm-frontend-task-card:nth-child(4) { animation-delay: 0.4s; }
            .hrm-frontend-task-card:nth-child(5) { animation-delay: 0.5s; }
            .hrm-frontend-task-card:nth-child(6) { animation-delay: 0.6s; }

            .hrm-frontend-task-card:hover {
                transform: translateY(-4px);
                box-shadow: 0 10px 15px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -2px rgba(0, 0, 0, 0.05);
                border-color: #0F766E;
            }

            /* Status Left Border */
            .hrm-frontend-task-card::before {
                content: '';
                position: absolute;
                left: 0;
                top: 0;
                bottom: 0;
                width: 4px;
                background: #E5E7EB;
                transition: background 0.3s;
            }
            .hrm-frontend-task-card.pending::before { background: #F59E0B; } /* Warning/Orange for Pending remains appropriate standard */
            .hrm-frontend-task-card.in_process::before { background: #3B82F6; } /* Blue for In Process */
            .hrm-frontend-task-card.completed::before { background: #0F766E; } /* Teal for Completed */

            .task-info strong {
                display: block;
                font-size: 18px;
                margin-bottom: 8px;
                color: #111827;
                font-weight: 700;
                transition: color 0.2s;
                word-wrap: break-word;
                overflow-wrap: break-word;
                word-break: break-word;
            }
            .hrm-frontend-task-card:hover .task-info strong {
                color: #0F766E;
            }

            .task-meta-badge {
                display: inline-flex;
                align-items: center;
                background: #F3F4F6;
                color: #4B5563;
                font-size: 12px;
                padding: 6px 12px;
                border-radius: 9999px;
                font-weight: 600;
                margin-right: 8px;
            }
            .task-meta-badge .dashicons {
                font-size: 14px;
                width: 14px;
                height: 14px;
                margin-right: 4px;
                color: #6B7280;
            }

            .hrm-btn-mark-complete {
                background: #0F766E;
                color: white;
                border: none;
                padding: 10px 20px;
                border-radius: 8px;
                font-weight: 600;
                font-size: 14px;
                cursor: pointer;
                transition: all 0.3s ease;
                box-shadow: 0 4px 6px -1px rgba(15, 118, 110, 0.3);
            }
            .hrm-btn-mark-complete:hover {
                background: #0D9488;
                transform: translateY(-2px);
                box-shadow: 0 6px 8px -1px rgba(15, 118, 110, 0.5);
            }

            .hrm-status-completed-badge {
                background: #F0FDF9; /* Light Teal */
                color: #0F766E; /* Teal Text */
                padding: 8px 16px;
                border-radius: 8px;
                font-weight: 600;
                font-size: 14px;
                letter-spacing: 0.5px;
                display: inline-flex;
                align-items: center;
                animation: pulse 2s infinite;
                border: 1px solid #CCFBF1;
            }
            @keyframes pulse {
                0% { box-shadow: 0 0 0 0 rgba(15, 118, 110, 0.2); }
                70% { box-shadow: 0 0 0 10px rgba(15, 118, 110, 0); }
                100% { box-shadow: 0 0 0 0 rgba(15, 118, 110, 0); }
            }

            /* Empty state */
            .hrm-empty-state {
                text-align: center;
                padding: 60px 20px;
                background: #F9FAFB;
                border-radius: 16px;
                border: 2px dashed #E5E7EB;
            }
            .hrm-empty-icon {
                font-size: 48px;
                width: 48px;
                height: 48px;
                color: #D1D5DB;
                margin-bottom: 20px;
            }

        </style>

        <div class="hrm-tasks-list">
            <?php if ( empty($tasks) && current_user_can('manage_options') ): ?>
                <div style="background:#fff3cd; color:#856404; padding:10px; border:1px solid #ffeeba; margin-bottom:15px; font-size:12px;">
                    <strong>Debug:</strong> No tasks found.<br>
                    Employee ID: <?php echo $employee_id; ?>
                </div>
            <?php endif; ?>
            
            <!-- Task Header with Summary -->
            <div class="hrm-task-header" style="margin-bottom: 24px;">
                <div style="display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap; gap: 16px;">
                    <div>
                        <h3 style="font-size: 24px; font-weight: 800; color: #111827; letter-spacing: -0.5px; margin: 0 0 8px 0;">My Tasks</h3>
                        <p style="color: #6B7280; font-size: 14px; margin: 0;">Track your ongoing and completed work assignments.</p>
                    </div>
                    <?php 
                    $total_tasks = count($tasks);
                    $ongoing_tasks = 0;
                    $completed_tasks = 0;
                    foreach($tasks as $t) {
                        if($t->status === 'completed') {
                            $completed_tasks++;
                        } else {
                            $ongoing_tasks++;
                        }
                    }
                    $completion_rate = $total_tasks > 0 ? round(($completed_tasks / $total_tasks) * 100) : 0;
                    ?>
                    <div class="hrm-task-summary" style="display: flex; gap: 16px;">
                        <div style="text-align: center; padding: 12px 20px; background: linear-gradient(135deg, #DBEAFE 0%, #EFF6FF 100%); border-radius: 12px; border: 1px solid #93C5FD;">
                            <div style="font-size: 24px; font-weight: 700; color: #1E40AF;"><?php echo $ongoing_tasks; ?></div>
                            <div style="font-size: 11px; font-weight: 600; color: #3B82F6; text-transform: uppercase;">Ongoing</div>
                        </div>
                        <div style="text-align: center; padding: 12px 20px; background: linear-gradient(135deg, #D1FAE5 0%, #ECFDF5 100%); border-radius: 12px; border: 1px solid #6EE7B7;">
                            <div style="font-size: 24px; font-weight: 700; color: #065F46;"><?php echo $completed_tasks; ?></div>
                            <div style="font-size: 11px; font-weight: 600; color: #10B981; text-transform: uppercase;">Completed</div>
                        </div>
                        <div style="text-align: center; padding: 12px 20px; background: linear-gradient(135deg, #F3F4F6 0%, #FAFAFA 100%); border-radius: 12px; border: 1px solid #E5E7EB;">
                            <div style="font-size: 24px; font-weight: 700; color: #374151;"><?php echo $completion_rate; ?>%</div>
                            <div style="font-size: 11px; font-weight: 600; color: #6B7280; text-transform: uppercase;">Done</div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Task Filter Tabs -->
            <div class="hrm-task-filter-tabs" style="display: flex; gap: 8px; margin-bottom: 24px; border-bottom: 2px solid #E5E7EB; padding-bottom: 0;">
                <?php 
                // Count tasks by status
                $ongoing_count = 0;
                $completed_count = 0;
                foreach($tasks as $t) {
                    if($t->status === 'completed') {
                        $completed_count++;
                    } else {
                        $ongoing_count++;
                    }
                }
                ?>
                <button type="button" class="hrm-task-tab active" data-filter="ongoing" style="
                    padding: 12px 20px;
                    border: none;
                    background: transparent;
                    font-size: 14px;
                    font-weight: 600;
                    color: #0F766E;
                    cursor: pointer;
                    border-bottom: 3px solid #0F766E;
                    margin-bottom: -2px;
                    transition: all 0.2s;
                    display: flex;
                    align-items: center;
                    gap: 8px;
                ">
                    <span class="dashicons dashicons-clock" style="font-size: 16px; width: 16px; height: 16px;"></span>
                    Ongoing Tasks
                    <span class="hrm-tab-count" style="background: #0F766E; color: white; padding: 2px 8px; border-radius: 12px; font-size: 11px; font-weight: 700;"><?php echo $ongoing_count; ?></span>
                </button>
                <button type="button" class="hrm-task-tab" data-filter="completed" style="
                    padding: 12px 20px;
                    border: none;
                    background: transparent;
                    font-size: 14px;
                    font-weight: 600;
                    color: #6B7280;
                    cursor: pointer;
                    border-bottom: 3px solid transparent;
                    margin-bottom: -2px;
                    transition: all 0.2s;
                    display: flex;
                    align-items: center;
                    gap: 8px;
                ">
                    <span class="dashicons dashicons-yes-alt" style="font-size: 16px; width: 16px; height: 16px;"></span>
                    Completed
                    <span class="hrm-tab-count" style="background: #E5E7EB; color: #374151; padding: 2px 8px; border-radius: 12px; font-size: 11px; font-weight: 700;"><?php echo $completed_count; ?></span>
                </button>
            </div>

            <!-- Ongoing Tasks Section -->
            <div class="hrm-task-section hrm-ongoing-tasks" id="ongoing-tasks-section">
                <div class="hrm-task-card-list">
                    <?php 
                    $has_ongoing = false;
                    if ( $tasks ) : foreach ( $tasks as $task ) : 
                        if($task->status === 'completed') continue;
                        $has_ongoing = true;
                        
                        // Determine status styling
                        $status_bg = '#FEF3C7'; // Default: Pending (Yellow)
                        $status_color = '#92400E';
                        $status_icon = '';
                        if($task->status === 'in_process') {
                            $status_bg = '#DBEAFE'; // Blue
                            $status_color = '#1E40AF';
                            $status_icon = '<span class="dashicons dashicons-update hrm-spin-slow" style="font-size: 12px; width: 12px; height: 12px; margin-right: 4px;"></span>';
                        }
                    ?>
                        <div class="hrm-frontend-task-card <?php echo $task->status; ?>">
                            <div class="task-info" style="flex: 1; padding-right: 24px; min-width: 0; overflow: hidden;">
                                <strong><?php echo esc_html( $task->title ); ?></strong>
                                <p style="color: #6B7280; font-size: 15px; margin: 0 0 16px; line-height: 1.6; max-width: 600px; word-wrap: break-word; overflow-wrap: break-word; word-break: break-word;"><?php echo esc_html( $task->description ); ?></p>
                                
                                <div style="display: flex; gap: 8px; flex-wrap: wrap;">
                                    <span class="task-meta-badge">
                                        <span class="dashicons dashicons-calendar-alt"></span>
                                        Due: <?php echo date('M d', strtotime($task->due_date)); ?>
                                    </span>
                                    <span class="task-meta-badge" style="background: <?php echo $status_bg; ?>; color: <?php echo $status_color; ?>;">
                                        <?php echo $status_icon . ucfirst(str_replace('_', ' ', $task->status)); ?>
                                    </span>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; endif; ?>
                    
                    <?php if(!$has_ongoing): ?>
                        <div class="hrm-empty-state" style="background: linear-gradient(135deg, #F0FDF9 0%, #ECFDF5 100%); border-color: #A7F3D0;">
                            <span class="dashicons dashicons-yes-alt hrm-empty-icon" style="color: #10B981;"></span>
                            <h4 style="margin: 0 0 8px 0; color: #065F46; font-weight: 700;">All caught up!</h4>
                            <p style="color: #6B7280; margin: 0;">You don't have any ongoing tasks. Great work!</p>
                        </div>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Completed Tasks Section -->
            <div class="hrm-task-section hrm-completed-tasks" id="completed-tasks-section" style="display: none;">
                <div class="hrm-task-card-list">
                    <?php 
                    $has_completed = false;
                    if ( $tasks ) : foreach ( $tasks as $task ) : 
                        if($task->status !== 'completed') continue;
                        $has_completed = true;
                    ?>
                        <div class="hrm-frontend-task-card completed" style="opacity: 0.9;">
                            <div class="task-info" style="flex: 1; padding-right: 24px; min-width: 0; overflow: hidden;">
                                <strong style="text-decoration: line-through; color: #6B7280;"><?php echo esc_html( $task->title ); ?></strong>
                                <p style="color: #9CA3AF; font-size: 15px; margin: 0 0 16px; line-height: 1.6; max-width: 600px; word-wrap: break-word; overflow-wrap: break-word; word-break: break-word;"><?php echo esc_html( $task->description ); ?></p>
                                
                                <div style="display: flex; gap: 8px; flex-wrap: wrap;">
                                    <span class="task-meta-badge" style="background: #F3F4F6; color: #6B7280;">
                                        <span class="dashicons dashicons-calendar-alt"></span>
                                        Due: <?php echo date('M d', strtotime($task->due_date)); ?>
                                    </span>
                                    <span class="task-meta-badge" style="background: #D1FAE5; color: #065F46;">
                                        <span class="dashicons dashicons-yes-alt" style="font-size: 12px; width: 12px; height: 12px; margin-right: 4px;"></span>
                                        Completed
                                    </span>
                                </div>
                            </div>
                            <div class="task-action" style="flex-shrink: 0; align-self: center;">
                                <span class="hrm-status-completed-badge">
                                    <span class="dashicons dashicons-yes-alt" style="margin-right: 6px;"></span> Done
                                </span>
                            </div>
                        </div>
                    <?php endforeach; endif; ?>
                    
                    <?php if(!$has_completed): ?>
                        <div class="hrm-empty-state">
                            <span class="dashicons dashicons-clipboard hrm-empty-icon"></span>
                            <h4 style="margin: 0 0 8px 0; color: #374151; font-weight: 700;">No completed tasks yet</h4>
                            <p style="color: #9CA3AF; margin: 0;">Complete your ongoing tasks and they'll appear here.</p>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        
        <!-- Tab Switching Script -->
        <script>
        jQuery(document).ready(function($) {
            // Tab switching for Ongoing / Completed
            $('.hrm-task-tab').on('click', function() {
                var filter = $(this).data('filter');
                
                // Update tab styles
                $('.hrm-task-tab').css({
                    'color': '#6B7280',
                    'border-bottom': '3px solid transparent'
                }).find('.hrm-tab-count').css({
                    'background': '#E5E7EB',
                    'color': '#374151'
                });
                
                $(this).css({
                    'color': '#0F766E',
                    'border-bottom': '3px solid #0F766E'
                }).find('.hrm-tab-count').css({
                    'background': '#0F766E',
                    'color': 'white'
                });
                
                // Show/hide sections with animation
                if(filter === 'ongoing') {
                    $('#completed-tasks-section').fadeOut(200, function() {
                        $('#ongoing-tasks-section').fadeIn(300);
                    });
                } else {
                    $('#ongoing-tasks-section').fadeOut(200, function() {
                        $('#completed-tasks-section').fadeIn(300);
                    });
                }
            });
        });
        </script>
        
        <!-- Additional Styles -->
        <style>
            @keyframes spin-slow {
                from { transform: rotate(0deg); }
                to { transform: rotate(360deg); }
            }
            .hrm-spin-slow {
                animation: spin-slow 2s linear infinite;
            }
            .hrm-task-tab:hover {
                color: #0F766E !important;
                background: #F0FDFA;
            }
            .hrm-frontend-task-card.completed {
                background: #FAFAFA;
            }
            .hrm-frontend-task-card.completed::before {
                background: #10B981 !important;
            }

        </style>

        <!-- Note: Task Add/Edit modal removed for employees - task management is admin-only -->
        <?php
    }

    public static function render_frontend_progress( $employee_id ) {
        global $wpdb;
        // Fetch tasks that are not completed (pending or in_process)
        $tasks = $wpdb->get_results( $wpdb->prepare( 
            "SELECT id, title, description, due_date, status FROM {$wpdb->prefix}hrm_tasks WHERE assigned_to = %d AND status != 'completed'", 
            $employee_id 
        ));
        
        // Fetch Today's Attendance Hours for Auto-Population
        $today_date = current_time( 'Y-m-d' );
        $attendance = $wpdb->get_row( $wpdb->prepare( 
            "SELECT hours_worked, check_in, check_out FROM {$wpdb->prefix}hrm_attendance WHERE employee_id = %d AND date = %s", 
            $employee_id, $today_date 
        ));
        
        // Calculate dynamic hours (if checked in but not out, use timer difference)
        $auto_hours = 0;
        if ( $attendance ) {
            if ( $attendance->hours_worked > 0 ) {
                $auto_hours = $attendance->hours_worked;
            } elseif ( $attendance->check_in && ! $attendance->check_out ) {
                // Approximate current hours
                $start = strtotime( get_gmt_from_date( $attendance->check_in ) );
                $now = time(); 
                $diff = $now - $start;
                $auto_hours = round( $diff / 3600, 2 );
            }
        }
        ?>
        <div class="hrm-progress-box" style="border: none; box-shadow: none; padding: 0;">
            <h3 style="font-size: 20px; font-weight: 700; color: #111827; margin-bottom: 24px;">Submit Daily Progress</h3>
            
            <?php if ( $auto_hours <= 0 ) : ?>
                 <div class="hrm-content-notice warning" style="margin-bottom: 24px;">
                    <span class="dashicons dashicons-warning" style="color: #F59E0B;"></span>
                    <strong>Note:</strong> You haven't recorded significant working hours today yet. Please Check In first.
                </div>
            <?php endif; ?>
            
            <?php if ( isset( $_GET['progress_submitted'] ) ) : ?>
                <div class="hrm-content-notice success" style="margin-bottom: 24px;">
                    <strong style="display:block; margin-bottom: 4px;">Report submitted successfully!</strong>
                </div>
            <?php endif; ?>
    
            <form method="post" class="hrm-form-grid" id="hrm-daily-progress-form">
                <?php wp_nonce_field( 'hrm_progress_action', 'hrm_progress_nonce' ); ?>
                <input type="hidden" name="hrm_action" value="submit_progress">
                <input type="hidden" name="employee_id" value="<?php echo $employee_id; ?>">
                
                <div class="hrm-form-group" style="margin-bottom: 24px;">
                    <label style="display: block; font-weight: 600; margin-bottom: 8px; font-size: 14px; color: #111827;">Date</label>
                    <input type="text" value="<?php echo current_time('Y-m-d'); ?>" readonly class="input-readonly" style="background: #F9FAFB; border: 1px solid #E5E7EB; color: #374151; font-weight: 500; border-radius: 8px; padding: 10px 14px; width: 100%;">
                </div>
    
                <div class="hrm-form-group full-width" style="margin-bottom: 24px;">
                    <div style="background: #F3F4F6; padding: 16px; border-radius: 8px; border: 1px solid #E5E7EB; margin-bottom: 20px;">
                        <label style="display: block; font-weight: 600; margin-bottom: 4px; font-size: 14px; color: #111827;">Total Hours Automatically Logged</label>
                        <div style="font-size: 24px; font-weight: 700; color: #059669;"><?php echo $auto_hours; ?> hrs</div>
                        <p style="font-size: 12px; color: #6B7280; margin: 4px 0 0;">This time will be distributed evenly across your selected tasks.</p>
                    </div>

                     <label style="display:block; margin-bottom:12px; font-weight: 600; font-size: 14px; color: #111827;">Tasks Worked On Today <span style="color:#EF4444;">*</span></label>
                     
                     <div class="hrm-task-checklist" style="border: 1px solid #E5E7EB; border-radius: 12px; background: #FAFAFA; overflow: hidden; margin-bottom: 20px;">
                        <?php if ( $tasks ) : foreach ( $tasks as $index => $task ) : 
                            $is_last = $index === count($tasks) - 1;
                            $current_status = isset($task->status) ? $task->status : 'pending';
                        ?>
                            <div class="hrm-task-item-wrapper" style="padding: 16px; background: white; <?php echo $is_last ? '' : 'border-bottom: 1px solid #F3F4F6;'; ?>">
                                <div class="hrm-checkbox-row" style="display:flex; align-items:center; gap:12px;">
                                    <input type="checkbox" name="task_entries[<?php echo $task->id; ?>][selected]" id="task_chk_<?php echo $task->id; ?>" class="hrm-task-checkbox" value="1" style="width: 18px; height: 18px; border-radius: 4px; border: 1px solid #D1D5DB; cursor: pointer;">
                                    <label for="task_chk_<?php echo $task->id; ?>" style="font-weight:600; cursor:pointer; font-size: 14px; color: #1F2937; user-select: none; flex: 1;">
                                        <?php echo esc_html( $task->title ); ?>
                                    </label>
                                    <span class="hrm-current-status-badge" style="font-size: 11px; padding: 4px 10px; border-radius: 20px; font-weight: 600; <?php 
                                        if($current_status === 'completed') {
                                            echo 'background: #D1FAE5; color: #065F46;';
                                        } elseif($current_status === 'in_process') {
                                            echo 'background: #DBEAFE; color: #1E40AF;';
                                        } else {
                                            echo 'background: #FEF3C7; color: #92400E;';
                                        }
                                    ?>">
                                        <?php echo ucfirst(str_replace('_', ' ', $current_status)); ?>
                                    </span>
                                </div>
                                <div class="hrm-task-details-inputs" id="details_<?php echo $task->id; ?>" style="display:none; padding-left: 30px; margin-top: 16px;">
                                    <!-- Work Description -->
                                    <div style="margin-bottom: 16px;">
                                        <label style="font-size:12px; font-weight: 500; color: #4B5563; display: block; margin-bottom: 4px;">Work Description <span style="color:#EF4444;">*</span></label>
                                        <textarea name="task_entries[<?php echo $task->id; ?>][desc]" rows="2" placeholder="Briefly describe what you did..." class="tiny-input" style="width: 100%; border-radius: 6px; border: 1px solid #E5E7EB; padding: 8px; resize: vertical;"></textarea>
                                    </div>
                                    
                                    <!-- Task Status Selection -->
                                    <div class="hrm-status-selector" style="margin-top: 12px;">
                                        <label style="font-size:12px; font-weight: 500; color: #4B5563; display: block; margin-bottom: 8px;">Task Status After Today <span style="color:#EF4444;">*</span></label>
                                        <div class="hrm-status-buttons" style="display: flex; gap: 12px;">
                                            <!-- Hidden input for form submission -->
                                            <input type="hidden" name="task_entries[<?php echo $task->id; ?>][task_status]" id="status_input_<?php echo $task->id; ?>" value="in_process">
                                            
                                            <!-- In Process Button -->
                                            <button type="button" class="hrm-status-btn hrm-status-in-process active" data-status="in_process" data-task-id="<?php echo $task->id; ?>" style="
                                                flex: 1;
                                                padding: 12px 16px;
                                                border-radius: 8px;
                                                border: 2px solid #3B82F6;
                                                background: linear-gradient(135deg, #EFF6FF 0%, #DBEAFE 100%);
                                                color: #1E40AF;
                                                font-weight: 600;
                                                font-size: 13px;
                                                cursor: pointer;
                                                display: flex;
                                                align-items: center;
                                                justify-content: center;
                                                gap: 8px;
                                                transition: all 0.25s ease;
                                                box-shadow: 0 2px 4px rgba(59, 130, 246, 0.15);
                                            ">
                                                <span class="dashicons dashicons-update" style="font-size: 16px; width: 16px; height: 16px;"></span>
                                                In Process
                                            </button>
                                            
                                            <!-- Completed Button -->
                                            <button type="button" class="hrm-status-btn hrm-status-completed" data-status="completed" data-task-id="<?php echo $task->id; ?>" style="
                                                flex: 1;
                                                padding: 12px 16px;
                                                border-radius: 8px;
                                                border: 2px solid #E5E7EB;
                                                background: #F9FAFB;
                                                color: #6B7280;
                                                font-weight: 600;
                                                font-size: 13px;
                                                cursor: pointer;
                                                display: flex;
                                                align-items: center;
                                                justify-content: center;
                                                gap: 8px;
                                                transition: all 0.25s ease;
                                            ">
                                                <span class="dashicons dashicons-yes-alt" style="font-size: 16px; width: 16px; height: 16px;"></span>
                                                Completed
                                            </button>
                                        </div>
                                        <p style="font-size: 11px; color: #9CA3AF; margin-top: 8px; font-style: italic;">
                                            <span class="dashicons dashicons-info" style="font-size: 12px; width: 12px; height: 12px; vertical-align: middle;"></span>
                                            This will automatically update the task status in the Tasks module.
                                        </p>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; else: ?>
                            <div style="padding: 20px; text-align: center; color: #6B7280;">No pending assigned tasks found.</div>
                        <?php endif; ?>
                     </div>
                     
                     <!-- General Progress Section -->
                     <div class="hrm-general-progress-wrapper" style="border: 1px solid #E5E7EB; border-radius: 12px; background: white; padding: 16px;">
                        <div class="hrm-checkbox-row" style="display:flex; align-items:center; gap:12px; margin-bottom: 12px;">
                            <input type="checkbox" id="chk_general_progress" class="hrm-general-checkbox" style="width: 18px; height: 18px; border-radius: 4px; border: 1px solid #D1D5DB; cursor: pointer;">
                            <label for="chk_general_progress" style="font-weight:600; cursor:pointer; font-size: 14px; color: #1F2937; user-select: none;">
                                General / Other Work
                            </label>
                        </div>
                        <div id="general_progress_details" style="display:none; padding-left: 30px;">
                             <div>
                                <label style="font-size:12px; font-weight: 500; color: #4B5563; display: block; margin-bottom: 4px;">Description <span style="color:#EF4444;">*</span></label>
                                <textarea name="general_progress" id="general_desc" rows="3" placeholder="Describe other work done today..." style="width: 100%; border-radius: 6px; border: 1px solid #E5E7EB; padding: 8px; resize: vertical;"></textarea>
                            </div>
                        </div>
                        </div>
                     </div>
                     
                     <small style="color:#6B7280; display:block; margin-top:8px; font-size: 13px;">Check tasks or 'General Work' to log your daily progress.</small>
                </div>
    
                <div class="hrm-form-actions">
                    <button type="submit" class="hrm-btn hrm-btn-primary" style="padding: 10px 24px; font-size: 14px;">Submit Report</button>
                </div>
            </form>
            
            <script>
            jQuery(document).ready(function($) {
                // Toggle details on checkbox
                $('.hrm-task-checkbox').on('change', function() {
                    var targetId = '#details_' + $(this).attr('id').split('_')[2];
                    if($(this).is(':checked')) {
                        $(targetId).slideDown();
                        $(targetId).find('textarea').prop('required', true);
                    } else {
                        $(targetId).slideUp();
                        $(targetId).find('textarea').prop('required', false);
                    }
                });

                // Toggle General Progress
                $('#chk_general_progress').on('change', function() {
                    if($(this).is(':checked')) {
                        $('#general_progress_details').slideDown();
                        $('#general_desc').prop('required', true);
                    } else {
                        $('#general_progress_details').slideUp();
                        $('#general_desc').prop('required', false);
                    }
                });

                // Status Button Toggle (In Process / Completed)
                $(document).on('click', '.hrm-status-btn', function(e) {
                    e.preventDefault();
                    var btn = $(this);
                    var taskId = btn.data('task-id');
                    var newStatus = btn.data('status');
                    var container = btn.closest('.hrm-status-buttons');
                    
                    // Update hidden input value
                    $('#status_input_' + taskId).val(newStatus);
                    
                    // Remove active state from all buttons in this group
                    container.find('.hrm-status-btn').removeClass('active');
                    
                    // Reset all buttons to inactive style
                    container.find('.hrm-status-in-process').css({
                        'border': '2px solid #E5E7EB',
                        'background': '#F9FAFB',
                        'color': '#6B7280',
                        'box-shadow': 'none'
                    });
                    container.find('.hrm-status-completed').css({
                        'border': '2px solid #E5E7EB',
                        'background': '#F9FAFB',
                        'color': '#6B7280',
                        'box-shadow': 'none'
                    });
                    
                    // Add active state to clicked button
                    btn.addClass('active');
                    
                    // Apply active styles based on status type
                    if(newStatus === 'in_process') {
                        btn.css({
                            'border': '2px solid #3B82F6',
                            'background': 'linear-gradient(135deg, #EFF6FF 0%, #DBEAFE 100%)',
                            'color': '#1E40AF',
                            'box-shadow': '0 2px 4px rgba(59, 130, 246, 0.15)'
                        });
                    } else if(newStatus === 'completed') {
                        btn.css({
                            'border': '2px solid #10B981',
                            'background': 'linear-gradient(135deg, #ECFDF5 0%, #D1FAE5 100%)',
                            'color': '#065F46',
                            'box-shadow': '0 2px 4px rgba(16, 185, 129, 0.15)'
                        });
                    }
                    
                    // Add a small pulse animation for feedback
                    btn.css('transform', 'scale(1.02)');
                    setTimeout(function() {
                        btn.css('transform', 'scale(1)');
                    }, 150);
                });

                // Validation
                $('#hrm-daily-progress-form').on('submit', function(e) {
                    var checkedTasks = $('.hrm-task-checkbox:checked').length;
                    var generalChecked = $('#chk_general_progress:checked').length;
                    
                    if(checkedTasks === 0 && generalChecked === 0) {
                        e.preventDefault();
                        alert('Please select at least one assigned task or fill out General Work progress.');
                        return false;
                    }
                    
                    // Validate that each checked task has a description
                    var valid = true;
                    $('.hrm-task-checkbox:checked').each(function() {
                        var taskId = $(this).attr('id').split('_')[2];
                        var desc = $('textarea[name="task_entries[' + taskId + '][desc]"]').val().trim();
                        if(!desc) {
                            valid = false;
                            $('textarea[name="task_entries[' + taskId + '][desc]"]').css('border-color', '#EF4444');
                        }
                    });
                    
                    if(!valid) {
                        e.preventDefault();
                        alert('Please provide a work description for all selected tasks.');
                        return false;
                    }
                });
            });
            </script>

            <!-- Daily Progress History List -->
            <div class="hrm-history-section" style="margin-top: 40px; padding-top: 24px; border-top: 1px solid #E5E7EB;">
                <h4 style="font-size: 18px; font-weight: 700; color: #111827; margin-bottom: 20px;">My Progress History</h4>
                
                <?php
                // Fetch last 10 entries for this employee
                $history = $wpdb->get_results( $wpdb->prepare(
                    "SELECT p.*, t.title as task_title 
                     FROM {$wpdb->prefix}hrm_daily_progress p 
                     LEFT JOIN {$wpdb->prefix}hrm_tasks t ON p.task_id = t.id 
                     WHERE p.employee_id = %d 
                     ORDER BY p.date DESC, p.created_at DESC 
                     LIMIT 10",
                    $employee_id
                ));
                ?>
                
                <div class="hrm-card" style="padding: 0; overflow: hidden; border: 1px solid #E5E7EB; border-radius: 12px;">
                    <table class="hrm-history-table" style="width: 100%; border-collapse: collapse;">
                        <thead style="background: #F9FAFB; border-bottom: 1px solid #E5E7EB;">
                            <tr>
                                <th style="padding: 12px 16px; text-align: left; font-size: 12px; font-weight: 600; color: #6B7280; text-transform: uppercase;">Date</th>
                                <th style="padding: 12px 16px; text-align: left; font-size: 12px; font-weight: 600; color: #6B7280; text-transform: uppercase;">Task</th>
                                <th style="padding: 12px 16px; text-align: left; font-size: 12px; font-weight: 600; color: #6B7280; text-transform: uppercase;">Hours</th>
                                <th style="padding: 12px 16px; text-align: left; font-size: 12px; font-weight: 600; color: #6B7280; text-transform: uppercase;">Description</th>
                                <th style="padding: 12px 16px; text-align: left; font-size: 12px; font-weight: 600; color: #6B7280; text-transform: uppercase;">Status</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if($history): foreach($history as $row): 
                                $status_badge = $row->status === 'completed' ? 'badge-green' : 'badge-blue';
                                $status_label = ucfirst(str_replace('_', ' ', $row->status));
                            ?>
                            <tr style="border-bottom: 1px solid #F3F4F6;">
                                <td style="padding: 16px; font-size: 14px; color: #374151;"><?php echo date('M d', strtotime($row->date)); ?></td>
                                <td style="padding: 16px; font-size: 14px; font-weight: 500; color: #111827;"><?php echo esc_html($row->task_title ?: 'General'); ?></td>
                                <td style="padding: 16px; font-size: 14px; font-weight: 600; color: #374151;"><?php echo $row->hours_logged; ?>h</td>
                                <td style="padding: 16px; font-size: 14px; color: #6B7280; line-height: 1.4;"><?php echo wp_trim_words($row->description, 8); ?></td>
                                <td style="padding: 16px;"><span class="hrm-badge <?php echo $status_badge; ?>" style="font-size: 11px; padding: 2px 8px;"><?php echo $status_label; ?></span></td>
                            </tr>
                            <?php endforeach; else: ?>
                            <tr>
                                <td colspan="5" style="padding: 32px; text-align: center; color: #9CA3AF;">No history found.</td>
                            </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        <?php
    }

    public static function render_daily_progress() {
        global $wpdb;
        $table_progress = $wpdb->prefix . 'hrm_daily_progress';
        $table_employees = $wpdb->posts;
        $table_tasks = $wpdb->prefix . 'hrm_tasks';

        // Filters
        $filter_date = isset($_GET['date']) ? sanitize_text_field($_GET['date']) : '';
        $filter_emp = isset($_GET['employee_id']) ? intval($_GET['employee_id']) : '';
        
        $where = "WHERE 1=1";
        if ( $filter_date ) $where .= $wpdb->prepare(" AND p.date = %s", $filter_date);
        if ( $filter_emp ) $where .= $wpdb->prepare(" AND p.employee_id = %d", $filter_emp);

        // Pagination
        $per_page = HRM_Pagination::get_per_page();
        $current_page = HRM_Pagination::get_current_page();
        $offset = HRM_Pagination::get_offset($current_page, $per_page);

        $sql = "SELECT p.*, e.post_title as employee_name, t.title as task_title 
                FROM $table_progress p
                LEFT JOIN $table_employees e ON p.employee_id = e.ID
                LEFT JOIN $table_tasks t ON p.task_id = t.id
                $where
                ORDER BY p.date DESC, p.created_at DESC
                LIMIT $offset, $per_page";
        
        $reports = $wpdb->get_results( $sql );
        
        $total_items = $wpdb->get_var( "SELECT COUNT(*) FROM $table_progress p $where" );

        $employees = get_posts(['post_type' => HRM_Setup::EMPLOYEE_POST_TYPE, 'numberposts' => -1]);
        ?>
        <div class="wrap hrm-wrap hrm-modern-dashboard">
            <div class="hrm-header-section">
                <div class="hrm-header-left">
                     <h1>Daily Progress</h1>
                     <p>Review daily work reports from your team.</p>
                </div>
            </div>

            <!-- Filter Bar -->
            <div class="hrm-filter-bar hrm-card" style="margin-bottom:24px;">
                <form method="get" style="display:flex; align-items:center; gap:12px; width:100%;">
                    <input type="hidden" name="page" value="hrm-daily-progress">
                    
                    <select name="employee_id" class="hrm-select-input">
                        <option value="">All Employees</option>
                        <?php foreach($employees as $emp): ?>
                            <option value="<?php echo $emp->ID; ?>" <?php selected($filter_emp, $emp->ID); ?>><?php echo esc_html($emp->post_title); ?></option>
                        <?php endforeach; ?>
                    </select>

                    <input type="date" name="date" value="<?php echo esc_attr($filter_date); ?>" class="hrm-select-input">
                    
                    <button type="submit" class="hrm-btn-primary">Filter</button>
                    
                    <?php if($filter_date || $filter_emp): ?>
                        <a href="<?php echo admin_url('admin.php?page=hrm-daily-progress'); ?>" class="hrm-btn-outline">Reset</a>
                    <?php endif; ?>
                </form>
            </div>

            <div class="hrm-card" style="padding:0; overflow:hidden;">
                <table class="hrm-table">
                    <thead>
                        <tr>
                            <th style="width: 20%;">Employee</th>
                            <th style="width: 10%;">Date</th>
                            <th style="width: 15%;">Task ID</th>
                            <th style="width: 5%;">Hours</th>
                            <th style="width: 30%;">Description</th>
                            <th style="width: 8%;">Status</th>
                            <th style="width: 12%;">Admin Comment</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if ( $reports ) : foreach ( $reports as $report ) : 
                             $status_badge = $report->status === 'completed' ? 'badge-green' : 'badge-blue';
                        ?>
                        <tr>
                            <td style="vertical-align: top;">
                                <div class="hrm-user-cell">
                                    <div class="hrm-avatar-sm"><?php echo strtoupper(substr($report->employee_name, 0, 1)); ?></div>
                                    <div style="display:flex; flex-direction:column;">
                                        <span style="font-weight:600;"><?php echo esc_html($report->employee_name); ?></span>
                                    </div>
                                </div>
                            </td>
                            <td class="text-muted" style="vertical-align: top; white-space: nowrap;"><?php echo date('M d, Y', strtotime($report->date)); ?></td>
                            <td style="vertical-align: top;">
                                <?php if($report->task_id > 0): ?>
                                    <a href="#" class="hrm-view-task-details hrm-task-id-badge" 
                                       data-progress-id="<?php echo $report->id; ?>" 
                                       data-task-id="<?php echo $report->task_id; ?>">
                                        TASK-<?php echo $report->task_id; ?>
                                    </a>
                                <?php else: ?>
                                    <span class="text-muted" style="font-size:13px; font-style:italic;">General Work</span>
                                <?php endif; ?>
                            </td>
                            <td style="vertical-align: top;"><strong><?php echo $report->hours_logged; ?></strong></td>
                            <td style="vertical-align: top;">
                                <div style="white-space: pre-wrap; word-wrap: break-word; font-size:13px; color: #4B5563; max-width: 400px;"><?php echo esc_html($report->description); ?></div>
                            </td>
                            <td style="vertical-align: top;"><span class="hrm-badge <?php echo $status_badge; ?>"><?php echo ucfirst(str_replace('_', ' ', $report->status)); ?></span></td>
                            <td style="vertical-align: top;">
                                <!-- Inline Comment Edit -->
                                <div class="hrm-admin-comment-box" style="width: 100%; min-width: 150px;">
                                    <textarea class="hrm-comment-input" data-id="<?php echo $report->id; ?>" placeholder="Add feedback..." rows="2" style="width:100%; min-height:60px;"><?php echo esc_textarea($report->admin_comment); ?></textarea>
                                    <button class="hrm-btn-icon hrm-save-comment" data-id="<?php echo $report->id; ?>" style="display:none; color:var(--primary-color);">
                                        <span class="dashicons dashicons-saved"></span>
                                    </button>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; else : ?>
                        <tr><td colspan="7" class="hrm-no-data" style="text-align:center; padding:30px;">No progress reports found.</td></tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>

            <!-- Pagination -->
            <?php 
            HRM_Pagination::render([
                'total_items' => $total_items,
                'current_page' => $current_page,
                'per_page' => $per_page,
                'query_args' => [
                    'page' => 'hrm-daily-progress',
                    'employee_id' => $filter_emp,
                    'date' => $filter_date
                ],
                'show_sizes' => true,
                'context' => 'admin'
            ]);
            HRM_Pagination::render_script();
            ?>

            <!-- Task Detail Modal -->
            <div id="hrm-task-detail-modal" class="hrm-modal" style="display:none;">
                <div class="hrm-modal-content hrm-premium-modal" style="max-width: 600px;">
                    <div class="hrm-modal-header-premium">
                        <div class="header-icon-wrapper" style="background:#EEF2FF; color:#4F46E5;">
                            <span class="dashicons dashicons-clipboard"></span>
                        </div>
                        <div class="header-text">
                            <h2 id="modal-task-title-header">Task Details</h2>
                            <p id="modal-task-id-display">Loading...</p>
                        </div>
                        <button class="hrm-modal-close-premium" onclick="closeModal('hrm-task-detail-modal')">
                             <span class="dashicons dashicons-no-alt"></span>
                        </button>
                    </div>
                    
                    <div class="hrm-modal-body-premium" id="hrm-modal-loader" style="padding:30px; text-align:center;">
                        <span class="dashicons dashicons-update hrm-spin" style="font-size:30px; width:30px; height:30px; color:#ccc;"></span>
                    </div>

                    <div class="hrm-modal-body-premium" id="hrm-modal-content" style="padding: 24px; display:none;">
                        <!-- Status Bar -->
                        <div style="display:flex; justify-content:space-between; align-items:center; margin-bottom:24px; background:#F9FAFB; padding:12px 16px; border-radius:8px;">
                             <span id="modal-status-badge"></span>
                             <span id="modal-date" style="font-weight:600; color:#374151;"></span>
                        </div>

                        <div class="hrm-detail-grid" style="display:grid; grid-template-columns: 1fr 1fr; gap:20px; margin-bottom:24px;">
                            <div class="detail-item">
                                <label style="display:block; font-size:11px; font-weight:700; color:#9CA3AF; text-transform:uppercase; margin-bottom:4px;">Assigned To</label>
                                <div id="modal-employee" style="font-weight:600; color:#1F2937;"></div>
                            </div>
                            <div class="detail-item">
                                <label style="display:block; font-size:11px; font-weight:700; color:#9CA3AF; text-transform:uppercase; margin-bottom:4px;">Hours Logged</label>
                                <div id="modal-hours" style="font-weight:600; color:#1F2937; font-size:18px;"></div>
                            </div>
                        </div>

                        <div style="margin-bottom:24px;">
                            <label style="display:block; font-size:11px; font-weight:700; color:#9CA3AF; text-transform:uppercase; margin-bottom:8px;">Task Description</label>
                            <div id="modal-task-desc" style="background:#F3F4F6; padding:12px; border-radius:6px; color:#4B5563; font-size:13px; line-height:1.5;"></div>
                        </div>

                        <div style="margin-bottom:24px;">
                            <label style="display:block; font-size:11px; font-weight:700; color:#9CA3AF; text-transform:uppercase; margin-bottom:8px;">Daily Progress Note</label>
                            <div id="modal-emp-comment" style="background:#F0FDF9; border:1px solid #D1FAE5; padding:12px; border-radius:6px; color:#065F46; font-size:14px; line-height:1.5;"></div>
                        </div>

                         <div>
                            <label style="display:block; font-size:11px; font-weight:700; color:#9CA3AF; text-transform:uppercase; margin-bottom:8px;">Admin Feedback</label>
                            <div id="modal-admin-comment" style="font-style:italic; color:#6B7280; font-size:13px;"></div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <style>
            .hrm-task-id-badge {
                display: inline-block;
                background: #EEF2FF;
                color: #4F46E5;
                padding: 4px 8px;
                border-radius: 4px;
                font-weight: 600;
                font-size: 12px;
                text-decoration: none;
                border: 1px solid #C7D2FE;
                transition: all 0.2s;
            }
            .hrm-task-id-badge:hover {
                background: #4F46E5;
                color: white;
                border-color: #4F46E5;
            }
            .hrm-premium-modal {
                background: white;
                border-radius: 12px;
                box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
                overflow: hidden;
            }
            .hrm-modal-header-premium {
                padding: 20px 24px;
                border-bottom: 1px solid #F3F4F6;
                display: flex;
                align-items: center;
                gap: 16px;
                position: relative;
            }
            .header-icon-wrapper {
                width: 40px;
                height: 40px;
                border-radius: 10px;
                display: flex;
                align-items: center;
                justify-content: center;
            }
            .header-text h2 { margin: 0; font-size: 18px; color: #111827; }
            .header-text p { margin: 2px 0 0; font-size: 13px; color: #6B7280; font-family: monospace; }
            .hrm-modal-close-premium {
                position: absolute;
                right: 20px;
                top: 50%;
                transform: translateY(-50%);
                background: transparent;
                border: none;
                cursor: pointer;
                color: #9CA3AF;
                padding: 8px;
                border-radius: 50%;
                display: flex; /* Fix alignment */
            }
            .hrm-modal-close-premium:hover { background: #F3F4F6; color: #4B5563; }
        </style>
        <style>
            .hrm-admin-comment-box { position: relative; }
            .hrm-comment-input {
                width: 100%;
                border: 1px solid transparent;
                background: transparent;
                padding: 4px;
                font-size: 13px;
                resize: vertical;
                min-height: 40px;
                transition: all 0.2s;
            }
            .hrm-comment-input:focus {
                background: #fff;
                border-color: #E5E7EB;
                box-shadow: 0 1px 2px rgba(0,0,0,0.05);
                outline: none;
            }
            .hrm-save-comment {
                position: absolute;
                bottom: 4px;
                right: 4px;
            }
        </style>
        <script>
            jQuery(document).ready(function($) {
                $('.hrm-comment-input').on('focus', function() {
                    $(this).next('.hrm-save-comment').show();
                });
                $('.hrm-save-comment').on('click', function() {
                    var btn = $(this);
                    var id = btn.data('id');
                    var comment = btn.prev('textarea').val();
                    
                    btn.html('<span class="dashicons dashicons-update" style="animation:spin 1s infinite linear"></span>');
                    
                    $.post(ajaxurl, {
                        action: 'hrm_update_task_status', // Reusing the handler that saves admin comments
                        progress_id: id,
                        comment: comment,
                        nonce: hrm_admin_params.nonce
                    }, function(res) {
                        if(res.success) {
                            btn.html('<span class="dashicons dashicons-yes" style="color:green"></span>');
                            setTimeout(function() { btn.hide().html('<span class="dashicons dashicons-saved"></span>'); }, 2000);
                        } else {
                            alert('Failed to save');
                        }
                    });
                });
            });
        </script>
        <?php
    }

}
